/*
 * Decompiled with CFR 0.152.
 */
package com.cdp4j.chromium;

import com.cdp4j.Constant;
import com.cdp4j.chromium.ChromiumChannel;
import com.cdp4j.chromium.ChromiumOptions;
import com.cdp4j.chromium.ChromiumVersion;
import com.cdp4j.exception.CdpException;
import com.cdp4j.logger.CdpLogger;
import com.cdp4j.logger.CdpLoggerFactory;
import com.cdp4j.serialization.GsonMapper;
import com.cdp4j.serialization.JacksonMapper;
import com.cdp4j.serialization.JsonMapper;
import java.io.ByteArrayInputStream;
import java.io.File;
import java.io.FileInputStream;
import java.io.IOException;
import java.net.ProxySelector;
import java.net.URI;
import java.net.http.HttpClient;
import java.net.http.HttpRequest;
import java.net.http.HttpResponse;
import java.nio.file.FileVisitResult;
import java.nio.file.Files;
import java.nio.file.LinkOption;
import java.nio.file.OpenOption;
import java.nio.file.Path;
import java.nio.file.SimpleFileVisitor;
import java.nio.file.attribute.BasicFileAttributes;
import java.nio.file.attribute.FileAttribute;
import java.time.LocalDate;
import java.time.format.DateTimeFormatter;
import java.time.temporal.ChronoUnit;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collections;
import java.util.EnumSet;
import java.util.List;
import java.util.Map;
import java.util.Optional;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.locks.LockSupport;
import java.util.stream.Collectors;

public class Chromium {
    private static final String FIELD_BRANCH_BASE_POSITION = "branch_base_position";
    private static final String FIELD_CURRENT_RELDATE = "current_reldate";
    private static final String FIELD_CURRENT_VERSION = "current_version";
    private static final String FIELD_CHANNEL = "channel";
    private static final String FIELD_VERSIONS = "versions";
    private static final String FIELD_OS = "os";
    private static final String CHANNEL_CANARY = "canary";
    private static final String CHANNEL_STABLE = "stable";
    private static final String CHANNEL_BETA = "beta";
    private static final String CHANNEL_DEV = "dev";
    private static final int HTTP_OK = 200;
    private static final int ZIP_EXTRACT_TIMEOUT = 60;
    private static final int MAX_RETRY = 100;
    private static final String CHROMIUM_DIRECTORY_NAME = "chromium-%s-%s";
    private static final String CHROMIUM_ARCHIVE_FILE = "chromium-%s-%s.zip";
    private static final String VERSION_INFO_FILE_EXTENSION = ".json";
    private static final String CHROMIUM_LATEST_VERSION_LIST_URI = "https://omahaproxy.appspot.com/all.json";
    private static final String WIN64_DOWNLOAD_LINK = "https://www.googleapis.com/download/storage/v1/b/chromium-browser-snapshots/o/Win_x64%%2F%d%%2Fchrome-win.zip?alt=media";
    private static final String LINUX_DOWNLOAD_LINK = "https://www.googleapis.com/download/storage/v1/b/chromium-browser-snapshots/o/Linux_x64%%2F%d%%2Fchrome-linux.zip?alt=media";
    private static final String MAC_DOWNLOAD_LINK = "https://www.googleapis.com/download/storage/v1/b/chromium-browser-snapshots/o/Mac%%2F%d%%2Fchrome-mac.zip?alt=media";
    private static final String MAC_ARM_DOWNLOAD_LINK = "https://www.googleapis.com/download/storage/v1/b/chromium-browser-snapshots/o/Mac_Arm%%2F%d%%2Fchrome-mac.zip?alt=media";
    private final JsonMapper mapper;
    private final String os;
    private final Path downloadPath;
    private final String downloadLink;
    private final CdpLogger log;
    private final ProxySelector proxySelector;
    private HttpClient client;

    public Chromium(ChromiumOptions o) {
        this.downloadPath = o.downloadPath();
        this.log = new CdpLoggerFactory(o.loggerType(), o.consoleLoggerLevel(), o.loggerNamePrefix()).getLogger("cdp.chromium");
        switch (o.jsonLibrary()) {
            case Jackson: {
                this.mapper = new JacksonMapper();
                break;
            }
            default: {
                this.mapper = new GsonMapper();
            }
        }
        if (Constant.WINDOWS) {
            this.os = "win64";
            this.downloadLink = WIN64_DOWNLOAD_LINK;
        } else if (Constant.MACOS) {
            String arch = System.getProperty("os.arch");
            if (arch != null && arch.contains("aarch")) {
                this.downloadLink = MAC_ARM_DOWNLOAD_LINK;
                this.os = "mac_arm64";
            } else {
                this.downloadLink = MAC_DOWNLOAD_LINK;
                this.os = "mac";
            }
        } else {
            this.downloadLink = LINUX_DOWNLOAD_LINK;
            this.os = "linux";
        }
        if (!Files.exists(this.downloadPath, new LinkOption[0])) {
            try {
                this.log.info("Creating Chromium download directory [{}]", this.downloadPath.toString());
                Files.createDirectory(this.downloadPath, new FileAttribute[0]);
            }
            catch (IOException e) {
                throw new CdpException(e);
            }
        }
        this.proxySelector = o.proxySelector();
    }

    public List<ChromiumVersion> listLatestVersions() {
        URI uri = URI.create(CHROMIUM_LATEST_VERSION_LIST_URI);
        HttpRequest request = HttpRequest.newBuilder(uri).GET().build();
        HttpResponse<byte[]> response = null;
        try {
            this.log.info("Listing latest Chromium versions (HTTP GET: [{}])", uri.toString());
            response = this.getHttpClient().send(request, HttpResponse.BodyHandlers.ofByteArray());
        }
        catch (IOException | InterruptedException e) {
            throw new CdpException(e);
        }
        byte[] content = response.body();
        if (this.log.isDebugEnabled()) {
            this.log.debug("Latest Chromium versions (HTTP Response): {}", new String(content));
        }
        List data = this.mapper.fromJson(new ByteArrayInputStream(content), List.class);
        ArrayList<ChromiumVersion> latestVersions = new ArrayList<ChromiumVersion>();
        DateTimeFormatter releaseDatePattern = DateTimeFormatter.ofPattern("MM/dd/yy");
        for (Map next : data) {
            if (!this.os.equals(next.get(FIELD_OS))) continue;
            List chromiumVersions = (List)next.get(FIELD_VERSIONS);
            for (Map chromiumVersion : chromiumVersions) {
                String channelStr = (String)chromiumVersion.get(FIELD_CHANNEL);
                ChromiumChannel channel = null;
                if (CHANNEL_DEV.equals(channelStr)) {
                    channel = ChromiumChannel.dev;
                } else if (CHANNEL_BETA.equals(channelStr)) {
                    channel = ChromiumChannel.beta;
                } else if (CHANNEL_STABLE.equals(channelStr)) {
                    channel = ChromiumChannel.stable;
                } else {
                    if (!CHANNEL_CANARY.equals(channelStr)) continue;
                    channel = ChromiumChannel.canary;
                }
                String releaseDateStr = (String)chromiumVersion.get(FIELD_CURRENT_RELDATE);
                LocalDate releaseDate = LocalDate.parse(releaseDateStr, releaseDatePattern);
                ChromiumVersion c = new ChromiumVersion((String)chromiumVersion.get(FIELD_CURRENT_VERSION), channel, releaseDate, Long.parseLong((String)chromiumVersion.get(FIELD_BRANCH_BASE_POSITION)));
                latestVersions.add(c);
            }
        }
        Collections.sort(latestVersions);
        Collections.sort(latestVersions, (o1, o2) -> Integer.compare(o2.getChannel().ordinal(), o1.getChannel().ordinal()));
        Collections.unmodifiableList(latestVersions);
        if (!latestVersions.isEmpty()) {
            this.log.info("Latest Chromium versions:", new Object[0]);
            for (ChromiumVersion v : latestVersions) {
                this.log.info("{}", v);
            }
        }
        return latestVersions;
    }

    public ChromiumVersion download(ChromiumChannel channel) {
        Path executablePath;
        ChromiumVersion installedChromium;
        List<ChromiumVersion> latestVersions = this.listLatestVersions();
        Optional<ChromiumVersion> found = latestVersions.stream().filter(c -> channel == c.getChannel()).findFirst();
        if (!found.isPresent()) {
            throw new CdpException("Chromium channel [" + (Object)((Object)channel) + "] not found");
        }
        ChromiumVersion latestChromium = found.get();
        List<ChromiumVersion> installedVersions = this.listInstalledVersions();
        Optional<ChromiumVersion> installed = installedVersions.stream().filter(c -> c.getChannel() == channel).findFirst();
        if (!installedVersions.isEmpty()) {
            this.log.info("Installed Chromium versions:", new Object[0]);
            for (ChromiumVersion v : installedVersions) {
                this.log.info("{}", v);
            }
        }
        if (!installed.isEmpty() && (installedChromium = installed.get()).equals(latestChromium) && (executablePath = this.getExecutablePath(installedChromium)) != null) {
            this.log.info("Skipping download, latest Chromium version is already installed: {}", latestChromium.toString());
            return latestChromium;
        }
        this.download(latestChromium);
        this.extract(latestChromium);
        return latestChromium;
    }

    public ChromiumVersion download(ChromiumChannel channel, long daysAgo) {
        ChromiumVersion c = this.isInstalled(channel, daysAgo);
        if (c != null) {
            this.log.info("Chromium is already installed: {}", c);
            return c;
        }
        return this.download(channel);
    }

    public Path getExecutablePath(ChromiumVersion c) {
        Path path = this.downloadPath.resolve(String.format(CHROMIUM_DIRECTORY_NAME, new Object[]{c.getVersion(), c.getChannel()}));
        path = Constant.WINDOWS ? path.resolve("chrome.exe") : (Constant.LINUX ? path.resolve("chrome") : path.resolve("Chromium.app").resolve("Contents").resolve("MacOS").resolve("Chromium"));
        try {
            if (Files.isExecutable(path)) {
                return path;
            }
        }
        catch (Throwable t) {
            this.log.error("Chromium executable not found: [{}]", path.toString());
            this.log.error(t.getMessage(), t);
            return null;
        }
        return null;
    }

    public List<ChromiumVersion> listInstalledVersions() {
        File[] files;
        ArrayList<ChromiumVersion> versions = new ArrayList<ChromiumVersion>();
        for (File next : files = this.downloadPath.toFile().listFiles((f, n) -> n.endsWith(VERSION_INFO_FILE_EXTENSION))) {
            try (FileInputStream is = new FileInputStream(next);){
                ChromiumVersion c = this.mapper.fromJson(is, ChromiumVersion.class);
                if (this.getExecutablePath(c) == null) continue;
                versions.add(c);
            }
            catch (IOException e) {
                throw new CdpException(e);
            }
        }
        Collections.sort(versions);
        Collections.sort(versions, (o1, o2) -> Integer.compare(o2.getChannel().ordinal(), o1.getChannel().ordinal()));
        Collections.unmodifiableList(versions);
        return versions;
    }

    public ChromiumVersion isInstalled(ChromiumChannel channel, long daysAgo) {
        ChromiumVersion c2;
        long diff;
        if (daysAgo < 0L) {
            throw new IllegalArgumentException("daysAgo");
        }
        ChromiumVersion found = null;
        Optional<ChromiumVersion> installedVersion = this.listInstalledVersions().stream().filter(c -> c.getChannel() == channel).findFirst();
        if (installedVersion.isPresent() && (diff = (c2 = installedVersion.get()).daysBetween(LocalDate.now())) <= daysAgo) {
            found = c2;
        }
        this.log.info("is Chromium [{}] version is already installed and released before at most [{}] days ago? [{}]", new Object[]{channel, daysAgo, found != null});
        return found;
    }

    private void download(ChromiumVersion c) {
        String chromiumArchiveFile;
        Path chromiumArchive;
        Artifact artifact = this.getDownloadUri(c);
        if (artifact.uri == null) {
            throw new CdpException("Invalid URI");
        }
        if (!Files.exists(this.downloadPath, new LinkOption[0])) {
            try {
                this.log.info("Creating Chromium download directory: [{}]", this.downloadPath.toString());
                Files.createDirectory(this.downloadPath, new FileAttribute[0]);
            }
            catch (IOException e) {
                throw new CdpException(e);
            }
        }
        if (Files.exists(chromiumArchive = this.downloadPath.resolve(chromiumArchiveFile = String.format(CHROMIUM_ARCHIVE_FILE, new Object[]{c.getVersion(), c.getChannel()})), new LinkOption[0])) {
            if (chromiumArchive.toFile().length() == artifact.length) {
                this.log.info("Skipping download, Chromium archive is already exist: [{}]", chromiumArchiveFile.toString());
                return;
            }
            try {
                this.log.info("Deleting existing Chromium archive file: [{}]. Expected file size: [{}], actual file size: [{}]", chromiumArchiveFile.toString(), artifact.length, Files.size(chromiumArchive));
                Files.delete(chromiumArchive);
            }
            catch (IOException e) {
                throw new CdpException(e);
            }
        }
        try {
            this.log.info("Downloading Chromium version: [{}], channel: [{}], release date: [{}]", new Object[]{c.getVersion(), c.getChannel(), c.getReleaseDate()});
            DownloadProgressTask progressTask = null;
            if (artifact.length > 0L) {
                progressTask = new DownloadProgressTask(chromiumArchive, artifact.length, this.log);
                Thread t = new Thread(progressTask);
                t.setDaemon(true);
                t.setName("CdpChromiumDownloadProgressTask");
                t.start();
            }
            HttpResponse<Path> response = this.getHttpClient().send(HttpRequest.newBuilder().GET().uri(artifact.uri).build(), HttpResponse.BodyHandlers.ofFile(chromiumArchive));
            if (progressTask != null) {
                progressTask.dispose();
            }
            if (response.statusCode() != 200) {
                throw new CdpException("Download error, HTTP status code: " + response.statusCode());
            }
            this.log.info("Chromium archive file downloaded successfully: [{}]", chromiumArchive.toString());
        }
        catch (IOException | InterruptedException e) {
            throw new CdpException(e);
        }
    }

    private Artifact getDownloadUri(ChromiumVersion version) {
        Artifact artifact = new Artifact();
        long position = version.getPosition();
        for (int count = 0; artifact.uri == null && count < 100; ++count) {
            String url = String.format(this.downloadLink, position);
            HttpRequest request = HttpRequest.newBuilder(URI.create(url)).method("HEAD", HttpRequest.BodyPublishers.noBody()).build();
            HttpResponse<Void> response = null;
            try {
                response = this.getHttpClient().send(request, HttpResponse.BodyHandlers.discarding());
                if (response.statusCode() == 200) {
                    artifact.uri = response.uri();
                    response.headers().firstValueAsLong("content-length").ifPresent(value -> artifact.length = value);
                    break;
                }
                --position;
                continue;
            }
            catch (IOException | InterruptedException e) {
                throw new CdpException(e);
            }
        }
        this.log.info("Chromium download uri: [{}]", artifact.uri);
        return artifact;
    }

    private Path extract(ChromiumVersion c) {
        Path chromiumArchive = this.downloadPath.resolve(String.format(CHROMIUM_ARCHIVE_FILE, new Object[]{c.getVersion(), c.getChannel()}));
        Path installPath = this.downloadPath.resolve(String.format(CHROMIUM_DIRECTORY_NAME, new Object[]{c.getVersion(), c.getChannel()}));
        try {
            if (Files.exists(installPath, new LinkOption[0])) {
                List files = Files.list(installPath).collect(Collectors.toList());
                if (!files.isEmpty()) {
                    return installPath;
                }
            } else {
                Files.createDirectory(installPath, new FileAttribute[0]);
            }
            String json = this.mapper.toJson(c);
            Path jsonFile = this.downloadPath.resolve(installPath.getFileName().toString() + VERSION_INFO_FILE_EXTENSION);
            if (Files.exists(jsonFile, new LinkOption[0])) {
                Files.delete(jsonFile);
            }
            Files.createFile(jsonFile, new FileAttribute[0]);
            Files.write(jsonFile, json.getBytes(), new OpenOption[0]);
        }
        catch (IOException e) {
            throw new CdpException(e);
        }
        try {
            ArrayList<String> args = new ArrayList<String>();
            if (Constant.WINDOWS) {
                args.add("cmd");
                args.add("/c");
            }
            if (Constant.LINUX) {
                args.add("unzip");
                args.add("-qq");
                args.add(chromiumArchive.getFileName().toString());
            } else {
                args.addAll(Arrays.asList("tar", "-xf", chromiumArchive.getFileName().toString(), "-C", installPath.getFileName().toString(), "--strip-components=1"));
            }
            this.log.info("Extracting Chromium archive file: [{}] to directory: [{}]", chromiumArchive.toString(), installPath.toString());
            ProcessBuilder pb = new ProcessBuilder(args);
            pb.directory(this.downloadPath.toFile());
            Process process = pb.start();
            process.waitFor(60L, TimeUnit.SECONDS);
            if (process.exitValue() != 0) {
                throw new CdpException("Unable to extract archive file: " + chromiumArchive.toString());
            }
            if (Files.exists(chromiumArchive, new LinkOption[0])) {
                Files.delete(chromiumArchive);
            }
            if (Constant.LINUX) {
                this.downloadPath.resolve("chrome-linux").toFile().renameTo(installPath.toFile());
            }
            return installPath;
        }
        catch (IOException | InterruptedException e) {
            throw new CdpException(e);
        }
    }

    private HttpClient getHttpClient() {
        if (this.client == null) {
            HttpClient.Builder builder = HttpClient.newBuilder().version(HttpClient.Version.HTTP_1_1).followRedirects(HttpClient.Redirect.ALWAYS);
            if (this.proxySelector != null) {
                builder.proxy(this.proxySelector);
            }
            this.client = builder.build();
        }
        return this.client;
    }

    public void deleteOldVersions(EnumSet<ChromiumChannel> channels, long daysAgo) {
        LocalDate releaseDate = LocalDate.now().minusDays(daysAgo);
        for (ChromiumVersion next : this.listInstalledVersions()) {
            Path installPath;
            long diff = ChronoUnit.DAYS.between(releaseDate, next.getReleaseDate());
            if (diff >= 0L || !channels.contains((Object)next.getChannel()) || !Files.exists(installPath = this.downloadPath.resolve(String.format(CHROMIUM_DIRECTORY_NAME, new Object[]{next.getVersion(), next.getChannel()})), new LinkOption[0]) || !Files.isDirectory(installPath, new LinkOption[0])) continue;
            try {
                this.log.info("Deleting old Chromium version, install path: [{}]", installPath.toString());
                Files.walkFileTree(installPath, new DirectoryCleanerVisitor(installPath));
                Path jsonMetaDataFile = this.downloadPath.resolve(installPath.getFileName().toString() + VERSION_INFO_FILE_EXTENSION);
                if (!Files.exists(jsonMetaDataFile, new LinkOption[0])) continue;
                Files.delete(jsonMetaDataFile);
            }
            catch (IOException e) {
                throw new CdpException(e);
            }
        }
    }

    private static class Artifact {
        public URI uri;
        private long length;

        private Artifact() {
        }
    }

    private static class DownloadProgressTask
    implements Runnable {
        private final Path archive;
        private final long expectedFileSize;
        private final long waitTime;
        private final CdpLogger log;
        private boolean done;
        private Thread currentThread;
        private long lastProgress;

        DownloadProgressTask(Path archive, long expectedFileSize, CdpLogger log) {
            this.archive = archive;
            this.expectedFileSize = expectedFileSize;
            this.log = log;
            this.waitTime = TimeUnit.MILLISECONDS.toNanos(250L);
        }

        @Override
        public void run() {
            this.currentThread = Thread.currentThread();
            while (!this.done) {
                LockSupport.parkNanos(this.waitTime);
                if (!Files.exists(this.archive, new LinkOption[0])) continue;
                try {
                    long actualFileSize = Files.size(this.archive);
                    if (actualFileSize <= 0L) continue;
                    long progress = Math.round((double)actualFileSize / (double)this.expectedFileSize * 100.0);
                    if (this.lastProgress != progress) {
                        this.log.info("Download progress: %{}", progress);
                        this.lastProgress = progress;
                    }
                    if (actualFileSize != this.expectedFileSize) continue;
                    this.done = true;
                }
                catch (IOException iOException) {}
            }
            this.currentThread = null;
        }

        public void dispose() {
            if (this.currentThread != null) {
                LockSupport.unpark(this.currentThread);
                this.done = true;
                this.currentThread = null;
            }
        }
    }

    private static class DirectoryCleanerVisitor
    extends SimpleFileVisitor<Path> {
        private final Path root;

        public DirectoryCleanerVisitor(Path root) {
            this.root = root;
        }

        @Override
        public FileVisitResult preVisitDirectory(Path dir, BasicFileAttributes attrs) throws IOException {
            if (dir.startsWith(this.root)) {
                return FileVisitResult.CONTINUE;
            }
            return FileVisitResult.SKIP_SUBTREE;
        }

        @Override
        public FileVisitResult visitFile(Path file, BasicFileAttributes attrs) throws IOException {
            Files.delete(file);
            return FileVisitResult.CONTINUE;
        }

        @Override
        public FileVisitResult postVisitDirectory(Path dir, IOException exc) throws IOException {
            if (dir.startsWith(this.root)) {
                Files.delete(dir);
            }
            return FileVisitResult.CONTINUE;
        }
    }
}

