// Use this file to customize the QF-Test HTML report
const modalTriggersSelector = 'a[href$=".png"], .message p, .longmessage p, .webresponse .step, .webrequest .step';
let modalTriggers = []
const triangleSvg = `<svg width="800px" height="800px" viewBox="0 0 24 24" fill="none" xmlns="http://www.w3.org/2000/svg"><path d="M10.285 3.85831C11.0618 2.56363 12.9382 2.56363 13.715 3.85831L22.1826 17.971C22.9824 19.3041 22.0222 21 20.4676 21H3.53238C1.97779 21 1.01757 19.3041 1.81739 17.971L10.285 3.85831Z" fill="currentColor"/></svg>`
const closeSvg = `<svg width="800px" height="800px" viewBox="0 0 24 24" fill="none" xmlns="http://www.w3.org/2000/svg"><path fill-rule="evenodd" clip-rule="evenodd" d="M12 1C5.92487 1 1 5.92487 1 12C1 18.0751 5.92487 23 12 23C18.0751 23 23 18.0751 23 12C23 5.92487 18.0751 1 12 1ZM15.7071 9.70711C16.0976 9.31658 16.0976 8.68342 15.7071 8.29289C15.3166 7.90237 14.6834 7.90237 14.2929 8.29289L12 10.5858L9.70711 8.29291C9.31658 7.90238 8.68342 7.90238 8.29289 8.29291C7.90237 8.68343 7.90237 9.3166 8.29289 9.70712L10.5858 12L8.2929 14.2929C7.90238 14.6834 7.90238 15.3166 8.2929 15.7071C8.68342 16.0976 9.31659 16.0976 9.70711 15.7071L12 13.4142L14.2929 15.7071C14.6834 16.0976 15.3166 16.0976 15.7071 15.7071C16.0976 15.3166 16.0976 14.6834 15.7071 14.2929L13.4142 12L15.7071 9.70711Z" fill="currentColor"/></svg>`
const copySvg = `<svg version="1.1" baseProfile="full" xmlns="http://www.w3.org/2000/svg" width="16" height="16" viewbox = "0 0 16 16"><rect x="5.75" y="1.75" width="7.5" height="9.5" style="stroke:#fff;stroke-width:0;fill:#fff;shape-rendering=geometricPrecision;" transform="rotate(0,9.5,6.5)"/><path d="M 7.2777778,11.5 H 11.722222 M 7.2777778,1.5 H 11.722222 M 13.5,3.3181818 V 9.6818182 M 5.5,3.3181818 V 9.6818182 M 11.722222,11.5 A 1.7777778,1.8181818 0 0 0 13.5,9.6818182 m -8,0 A 1.7777778,1.8181818 0 0 0 7.2777778,11.5 M 13.5,3.3181818 A 1.7777778,1.8181818 0 0 0 11.722222,1.5 m -4.4444442,0 A 1.7777778,1.8181818 0 0 0 5.5,3.3181818" style="stroke:#808080; stroke-linejoin:round; stroke-linecap:round; fill:none; stroke-width:1" transform="rotate(0,8.0,13.5)"/><rect x="2.75" y="4.75" width="7.5" height="9.5" style="stroke:#fff;stroke-width:0;fill:#fff;shape-rendering=geometricPrecision;" transform="rotate(0,6.5,9.5)"/><path d="M 4.2777778,14.5 H 8.7222222 M 4.2777778,4.5 H 8.7222222 M 10.5,6.3181818 V 12.681818 M 2.5,6.3181818 V 12.681818 M 8.7222222,14.5 A 1.7777778,1.8181818 0 0 0 10.5,12.681818 m -8,0 A 1.7777778,1.8181818 0 0 0 4.2777778,14.5 M 10.5,6.3181818 A 1.7777778,1.8181818 0 0 0 8.7222222,4.5 m -4.4444444,0 A 1.7777778,1.8181818 0 0 0 2.5,6.3181818" style="stroke:#808080; stroke-linejoin:round; stroke-linecap:round; fill:none; stroke-width:1" transform="rotate(0,8.0,13.5)"/></svg>`

const localizedStrings = {
    "Previous row (arrow up)": "Vorherige Zeile (Pfeiltaste hoch)",
    "Next row (arrow down)": "Nächste Zeile (Pfeiltaste runter)",
    "Previous (Arrow left)": "Zurück (Pfeiltaste links)",
    "Next (Arrow right)": "Weiter (Pfeiltaste rechts)",
    "Close (escape)": "Schließen (ESC-Taste)",
    "Copy to clipboard": "In Zwischenablage kopieren",
}
document.addEventListener("DOMContentLoaded", function(event) {
    // Attach simple modals to image links and messages
    modalTriggers = Array.from(document.querySelectorAll(modalTriggersSelector));
    let selectedTriggerId = null
    if (window.location.search) {
        // Open linked modal
        selectedTriggerId = window.location.search.replace("?modal=", "");
    }
    // Doing this without the timeouts can create huge browser slowdown
    setTimeout(() => {
        for (const [i, trigger] of modalTriggers.entries()) {
            trigger.dataset.modalTriggerId = "t"+i;
            const prevTrigger = modalTriggers?.[modalTriggers.indexOf(trigger) - 1];
            const nextTrigger = modalTriggers?.[modalTriggers.indexOf(trigger) + 1];
            attachModal(trigger, prevTrigger, nextTrigger);
            if (selectedTriggerId == trigger.dataset.modalTriggerId) {
                if (trigger) {
                    setTimeout(() => { trigger.click(); }, 0);
                }
            }
        }
    }, 0);

    // Modal Keyboard navigation
    document.body.addEventListener('keydown', (keyEvent) => {
        modal = document.querySelector("#modal");
        if (!modal || ! modal.open) {
            return;
        }
        if (keyEvent.ctrlKey || keyEvent.shiftKey || keyEvent.altKey || keyEvent.metaKey) {
            return;
        }

        let buttonId = null;
        switch (keyEvent.key) {
            case "Escape": buttonId = "modal-close"; break;
            case "ArrowLeft": buttonId = "modal-prev"; break;
            case "ArrowRight": buttonId = "modal-next"; break;
            case "ArrowUp": buttonId = "modal-previous-row"; break;
            case "ArrowDown": buttonId = "modal-next-row"; break;
        }
        if (buttonId) {
            let button = document.querySelector("button#"+buttonId);
            if (button) {
                button.click();
                keyEvent.stopPropagation();
            }
        }
    });

    window.addEventListener('resize', () => {
        markUndersized(document.querySelector(".mainImg"))
    });

    document.querySelectorAll(".webresponse p.nowrap, .webrequest p.nowrap, .multi-item-wrapper p.nowrap").forEach((el) => {
        setTimeout(() => {
            addSyntaxHighlighting(el)
            addCopyToClipboard(el)
        }, 0)
    })
});

/**
 * Attach a modal overlay to the given trigger, with next/previous navigation
 *
 * @param {Element} trigger
 * @param {Element} prevTrigger
 * @param {Element} nextTrigger
 * @returns {Element}
 */
function attachModal(trigger, prevTrigger, nextTrigger) {
    const lastDown = {};
    trigger.addEventListener("mousedown", (downEvent) => {
        lastDown.x = downEvent.clientX;
        lastDown.y = downEvent.clientY;
    });
    trigger.addEventListener("click", (clickEvent) => {
        if (clickEvent.target.closest("button")) {
            return; // dont trigger modal on "copy to clipboard"
        }
        if (trigger.tagName == "P" || trigger.querySelector("p")) {
            const realClick = clickEvent.clientX > 0 || clickEvent.clientY > 0;
            if (realClick && (Math.abs(clickEvent.clientX - lastDown.x) > 5 || Math.abs(clickEvent.clientY - lastDown.y) > 5)) {
                return; // don't open modal on message select
            }
        }

        clickEvent.preventDefault();
        let modal = document.querySelector("DIALOG#modal");
        if (!modal) {
            modal = document.createElement("DIALOG");
            modal.id = "modal";
            document.body.appendChild(modal);
        }
        modal.dataset.trigger = trigger.dataset.modalTriggerId;
        if (prevTrigger) {
            modal.dataset.prevTrigger = prevTrigger.dataset.modalTriggerId;
        }
        if (nextTrigger) {
            modal.dataset.nextTrigger = nextTrigger.dataset.modalTriggerId;
        }
        modal.innerHTML = "";
        modal.appendChild(modalContent(trigger, prevTrigger, nextTrigger, modal));

        trigger.focus();
        modal.addEventListener("wheel", (wheelEvent) => {
            wheelEvent.preventDefault();
        });
        modal.showModal();

        if (window.location.search) {
            history.replaceState(null, null, (window.location+"").replace(window.location.search, "?modal="+trigger.dataset.modalTriggerId));
        } else if (window.location.hash) {
            history.replaceState(null, null, (window.location+"").replace("#", "?modal="+trigger.dataset.modalTriggerId+"#"));
        } else {
            history.replaceState(null, null, window.location+"?modal="+trigger.dataset.modalTriggerId);
        }
    });
}

function switchModal(modal, nextModalTrigger) {
    nextModalTrigger.click();
    // Scroll the current trigger link into view
    navigationBarOffset = 85;
    window.scrollTo({
        top: nextModalTrigger.getBoundingClientRect().top + window.pageYOffset - navigationBarOffset,
        behavior: "smooth"
    });
    // Give the current trigger link a
    // focus ring to orientate the user.
    nextModalTrigger.focus();
}

/**
 *
 * @param {Element} trigger
 * @param {Element} prevTrigger
 * @param {Element} nextTrigger
 * @param {Element} modal
 * @returns {Element}
 */
function modalContent(trigger, prevTrigger, nextTrigger, modal) {

    let modalContent = document.createElement("DIV");
    modalContent.id = "modal-content";
    const imgWrapper = createModalContentFor(trigger);
    modalContent.appendChild(imgWrapper);

    modalContent.onclick = (event) => {
        if (!modalContent.querySelector("img:not(.icon)")) return;
        if (event.target.closest(".modal-content-desc")) return;

        document.body.classList.toggle("modal-img-pixel-correct");
        markUndersized(imgWrapper)
    };

    const modalOpening = modal && modal.getBoundingClientRect().height == 0;
    setTimeout(() => markUndersized(imgWrapper),modalOpening ? 210 : 0);

    let parentRow = getLogicalParentRow(trigger);
    let relatedTriggers = getRelatedTriggers(trigger, parentRow);

    let close = document.createElement("BUTTON");
    close.id = "modal-close";
    close.innerHTML = closeSvg;
    close.setAttribute("autofocus", null);
    close.title = _("Close (escape)");
    close.onclick = () => {
        closeModal(modal);
    }

    let header = document.createElement("DIV");
    header.id = "modal-header";
    header.appendChild(createNavigateRowButton(relatedTriggers, prevTrigger, true));
    let parentEl = trigger.closest(".leaf, .branch");
    if (parentEl) {
        let heading = parentEl.querySelector("h2,h3,h4,h5,h6");
        if (heading) {
            let icon = heading.querySelector("img");
            if (icon) {
                header.appendChild(icon.cloneNode(false));
            }
            let title = document.createElement("DIV")
            title.id = "modal-table-heading"
            title.innerText = heading.innerText
            header.appendChild(title);
        }
    }
    if (parentRow) {
        let icon = parentRow.querySelector("td.marginicon img");
        let titleicon = parentRow.querySelector("td.testset img.icon, td.testcase img.icon, td.teststep img.icon");
        let title = parentRow.querySelector("td.testset, td.testcase, td.teststep");
        if (header.querySelector("#modal-table-heading") && (icon || titleicon || title)) {
            let separator = document.createElement("DIV")
            separator.classList.add("breadcrumb-separator")
            separator.innerHTML = "&#10132;"
            header.appendChild(separator);
        }
        if (icon) {
            header.appendChild(icon.cloneNode(false));
        }
        if (titleicon) {
            header.appendChild(titleicon.cloneNode(false));
        }
        if (title) {
            let el = document.createElement("DIV");
            el.classList.add("title");
            el.textContent = title.textContent;
            header.appendChild(el);
        }
    }
    header.appendChild(createNavigateRowButton(relatedTriggers, nextTrigger));
    header.appendChild(close);


    let footerItems = []

    if (parentRow) {
        if (relatedTriggers) {
            footerItems = relatedTriggers.map((targetTrigger) => {
                let button = document.createElement("BUTTON");
                button.classList.add("relatedTrigger");
                button.appendChild(createModalPreviewContentFor(targetTrigger, true));
                if (trigger === targetTrigger) {
                    button.classList.add("current");
                } else {
                    button.onclick = () => {
                        switchModal(modal, targetTrigger);
                    };
                }
                return button;
            });
        }
    }

    let footer = document.createElement("DIV");
    footer.id = "modal-footer";
    if (footerItems) {
        footerItems.forEach((item) => {
            footer.appendChild(item);
        });
    }

    let container = document.createElement("DIV");
    container.id = "modal-container";

    container.appendChild(header);
    container.appendChild(modalContent);
    container.appendChild(createNavigateTriggerButton(prevTrigger, true));
    container.appendChild(footer);
    container.appendChild(createNavigateTriggerButton(nextTrigger));

    return container;
}

function getRelatedTriggers(trigger, parentRow) {
    if (!parentRow) {
        parentRow = getLogicalParentRow(trigger)
    }
    let result = [];
    if (parentRow) {
        result = Array.from(parentRow.querySelectorAll(modalTriggersSelector));
    }
    if (result.length == 0) {
        /* If the following rows are logical siblings of the current row, we can fetch their triggers. */
        let triggerIndentation = getRowIndentation(trigger.closest("tr"))
        if (triggerIndentation) {
            let candidateRow = parentRow.nextElementSibling
            while (candidateRow) {
                indentation = getRowIndentation(candidateRow)
                if (indentation == triggerIndentation) {
                    result = result.concat(Array.from(candidateRow.querySelectorAll(modalTriggersSelector)));
                } else {
                    break
                }
                candidateRow = candidateRow.nextElementSibling
            }
        }
    }
    if (result.length > 10) {
        /* These are too many results, so instead fall back to fetching the related triggers just in the concrete row we are in. */
        console.log("trigger.closest('tr')", trigger.closest('tr'))
        result = Array.from(trigger.closest('tr').querySelectorAll(modalTriggersSelector));
    }
    return result;
}

function createNavigateTriggerButton(targetTrigger, back) {
    let result = document.createElement("BUTTON");
    result.innerHTML = triangleSvg;
    if (back) {
        result.id = "modal-prev";
        result.title = _("Previous (Arrow left)");
    } else {
        result.id = "modal-next";
        result.title = _("Next (Arrow right)");
    }
    if (targetTrigger) {
        result.onclick = () => {
            switchModal(modal, targetTrigger);
        }
    } else {
        result.setAttribute("disabled", 1);
    }
    return result;

}

function createNavigateRowButton(relatedTriggers, fallbackTrigger, back) {
    let result = document.createElement("BUTTON");
    result.id = back ? "modal-previous-row" : "modal-next-row";
    result.innerHTML = triangleSvg;
    result.title = back ? _("Previous row (arrow up)") : _("Next row (arrow down)");

    let targetRowTrigger = null
    let bestRelatedTrigger = relatedTriggers.at(back ? 0 : -1);
    if (bestRelatedTrigger) {
        targetRowTrigger = modalTriggers?.at(modalTriggers.indexOf(bestRelatedTrigger) + (back ? -1 : 1))
        if (back) {
            targetRowTrigger = getRelatedTriggers(targetRowTrigger).at(0)
        }
    }
    if (!targetRowTrigger && fallbackTrigger) {
        targetRowTrigger = fallbackTrigger
    }
    if (targetRowTrigger) {
        result.onclick = () => { switchModal(modal, targetRowTrigger); };
    } else {
        result.setAttribute("disabled", 1);
    }
    return result;
}

function createModalContentFor(trigger) {
    if (trigger.tagName == "A") {
        let thumbnail = trigger.querySelector("img")
        let imgTitle = (trigger.textContent || !thumbnail) ? trigger.textContent : thumbnail.getAttribute("title");
        let imgUrl = trigger.getAttribute("href");
        let imgWrapper = document.createElement("DIV");
        imgWrapper.classList.add("imgWrapper");
        imgWrapper.innerHTML = `
        <div class="modal-content-desc"><strong>${imgTitle}</strong> (${imgUrl})</div>
        <div class="inner-img-wrapper">
            <img alt="${imgTitle}" src="${imgUrl}" />
        </div>
        `;
        imgWrapper.classList.add("mainImg");
        return imgWrapper;
    } else if (trigger.tagName == "P") {
        message = document.createElement("P");
        message.classList = trigger.classList;
        message.innerHTML = trigger.innerHTML;
        addCopyToClipboard(message)
        return message;
    } else {
        div = document.createElement(trigger.tagName);
        div.setAttribute("class", "multi-item-wrapper")
        div.innerHTML = trigger.innerHTML;
        div.querySelectorAll("p").forEach((el) => {
            if (!el.querySelector(".syntax-code")) {
                addSyntaxHighlighting(el)
            }
            addCopyToClipboard(el)
        })
        return div;
    }
}
function createModalPreviewContentFor(trigger) {
    if (trigger.tagName == "A") {
        let thumbnail = trigger.querySelector("img")
        let imgTitle = (trigger.textContent || !thumbnail) ? trigger.textContent : thumbnail.getAttribute("title");
        // Use smaller thumbnail image for previews
        let imgUrl = thumbnail ? thumbnail.getAttribute("src") : trigger.getAttribute("href");
        let imgWrapper = document.createElement("DIV");
        imgWrapper.classList.add("imgWrapper");
        imgWrapper.innerHTML = `
        <div class="modal-content-desc"><strong>${imgTitle}</strong> (${imgUrl})</div>
        <div class="inner-img-wrapper">
            <img alt="${imgTitle}" src="${imgUrl}" />
        </div>
        `;
        return imgWrapper;
    }
    let div = document.createElement(trigger.tagName);
    div.innerHTML = trigger.innerHTML;
    div.querySelectorAll("p").forEach((p => {
        p.innerHTML = p.innerText.substring(0, 200)+'…'
    }))
    return div;
}

function markUndersized(imgWrapper) {
    if (! imgWrapper) return;
    const img = imgWrapper.querySelector("img:not(.icon)");
    if (! img) return;
    const rect = imgWrapper.parentElement.getBoundingClientRect();
    const desc = imgWrapper.querySelector(".modal-content-desc");
    if (desc) {
        rect.height -= desc.getBoundingClientRect().height;
    }
    const percentage = Math.min(rect.width/img.naturalWidth, rect.height/img.naturalHeight);
    if (percentage > 1) {
        img.classList.add("undersized");
    } else {
        img.classList.remove("undersized");
    }
    if (percentage >= 4) {
        img.classList.add("pixelated");
    } else {
        img.classList.remove("pixelated");
    }
}

function closeModal(modal) {
    modal.close()
    if (window.location.search) {
        history.replaceState(null, null, (window.location+"").replace(window.location.search, ""));
    }
}

function addSyntaxHighlighting(el) {
    let fullCode = el.innerHTML;
    let code;
    let unprocessedCodeSuffix;
    let processingLimit = 10000;
    if (fullCode.length > processingLimit) {
        code = fullCode.substring(0, processingLimit);
        unprocessedCodeSuffix = fullCode.substring(processingLimit);
    } else {
        code = fullCode;
        unprocessedCodeSuffix = "";
    }
    // one big regex with groups for the different types of tokens.
    // This is not very precise but better than nothing.
    let syntaxRegex
    let syntax
    if (code.trim().startsWith("{") || code.trim().startsWith("[")) {
        // JavaScript
        syntax = "js"
        syntaxRegex = /(\b(true|false|null)\b)|([\{|\}|\[|\]|:|\\.])|[^\\]?("[^"]*?[^\\]"):|[^\\]?(".*?[^\\]")/g;
    } else if (code.trim().startsWith("&lt;")) {
        // XML
        syntax = "xml"
        syntaxRegex = /((&amp;.*?;))|([^\s]+=)|(&lt;.*?(?:\s|&gt;))|(".*?")/g;
    } else {
        return
    }

    code = code.replace(syntaxRegex, function
    (match, keyword, keywordGroup, operator, key, literal) {
        if (keyword) {
            return `<span class="syntax-keyword">${keyword}</span>`;
        } else if (operator) {
            return `<span class="syntax-operator">${operator}</span>`;
        } else if (key) {
            const operatorSuffix = (syntax == "js" ? ":" : "")
            return `<span class="syntax-key">${key}</span>`+ (operatorSuffix ? `<span class="syntax-operator">${operatorSuffix}</span>` : '')
        } else if (literal) {
            return `<span class="syntax-literal">${literal}</span>`;
        }
        return match;
    });

    el.innerHTML = `<span class="syntax-code">${code}${unprocessedCodeSuffix}</span>`;

    if (syntax == "js") {
        el.querySelectorAll(".syntax-literal").forEach((literalEl) => {
            literalEl.innerHTML = literalEl.innerHTML.replace(/\\./g, (match) => {
                return `<span class="syntax-escapesequence">${match}</span>`
            })
        })
    }
}

function addCopyToClipboard(el) {
    el.querySelectorAll("button.copy-to-clipboard").forEach((b)=>{b.remove()})
    let copyButton = document.createElement('button');
    copyButton.innerHTML = copySvg
    copyButton.title = _("Copy to clipboard")
    copyButton.classList.add("copy-to-clipboard")
    copyButton.onclick = (e) => {
        navigator.clipboard.writeText(el.innerText);
        copyButton.classList.add("success")
        copyButton.blur()
        setTimeout(() => {
            copyButton.classList.remove("success")
        }, 1000)

    }
    el.lastChild.after(copyButton)
}

/**
 * Returns the next logical parent row of the current item, otherwise the row itself.
 *
 * The HTML table structures of the report sometimes have rows that are logical children of a row above, marked visually
 * via indentation.
 **/
function getLogicalParentRow(element) {
    // The parentRow will be shown in the modal header. We first check if there is a
    let targetRowIndentation = getRowIndentation(element.closest("tr"))
    if (targetRowIndentation) {
        /* This is a little tedious since this logical hierarchy is only encoded as padding in the HTML */
        let candidateRow = element.closest('tr').previousElementSibling
        while (candidateRow) {
            let candidateIndentation = getRowIndentation(candidateRow);
            if (candidateIndentation) {
                if (candidateIndentation < targetRowIndentation) {
                    return candidateRow;
                } else {
                    candidateRow = candidateRow.previousElementSibling
                }
            } else {
                break
            }
        }
    }

    return element.closest("tr");
}

/**
 * Calculate the indentation of the given row. Will return 0 if the row does not exist.
 *
 * Relies on both the css class and the style attribute being set on indented items!
 */
function getRowIndentation(row) {
    if (!row) {
        return 0;
    }
    let paddedElement = row.querySelector("[class*=padding-left-]");
    if (paddedElement) {
        return paddedElement.style.paddingLeft
    } else {
        return 0
    }
}

function _(string) {
    if (document.documentElement.lang === "de") {
        translation = localizedStrings[string]
        if (!translation) {
            console.error("No DE translation found for string", string)
            return string
        }
        return translation
    }
    return string
}