/*
 * Decompiled with CFR 0.152.
 */
package com.influxdb.client;

import com.influxdb.LogLevel;
import com.influxdb.client.domain.WriteConsistency;
import com.influxdb.client.domain.WritePrecision;
import com.influxdb.client.write.PointSettings;
import com.influxdb.client.write.WriteParameters;
import com.influxdb.exceptions.InfluxException;
import com.influxdb.utils.Arguments;
import java.io.IOException;
import java.io.InputStream;
import java.time.Duration;
import java.time.temporal.ChronoUnit;
import java.util.Collections;
import java.util.Properties;
import java.util.regex.Matcher;
import java.util.regex.Pattern;
import javax.annotation.Nonnull;
import javax.annotation.Nullable;
import javax.annotation.concurrent.NotThreadSafe;
import okhttp3.HttpUrl;
import okhttp3.OkHttpClient;
import okhttp3.Protocol;

public final class InfluxDBClientOptions {
    private static final Pattern TAGS_PROPERTY = Pattern.compile("(influx2\\.tags\\.)(.+)");
    private static final Pattern DURATION_PATTERN = Pattern.compile("^(\\d+)([a-zA-Z]{0,2})$");
    private final String url;
    private final String clientType;
    private final OkHttpClient.Builder okHttpClient;
    private final LogLevel logLevel;
    private final AuthScheme authScheme;
    private final char[] token;
    private final String username;
    private final char[] password;
    private final String org;
    private final String bucket;
    private final WritePrecision precision;
    private final WriteConsistency consistency;
    private final PointSettings pointSettings;

    private InfluxDBClientOptions(@Nonnull Builder builder) {
        Arguments.checkNotNull(builder, "InfluxDBClientOptions.Builder");
        this.url = builder.url;
        this.clientType = builder.clientType;
        this.okHttpClient = builder.okHttpClient;
        this.logLevel = builder.logLevel;
        this.authScheme = builder.authScheme;
        this.token = builder.token;
        this.username = builder.username;
        this.password = builder.password;
        this.org = builder.org;
        this.bucket = builder.bucket;
        this.precision = builder.precision != null ? builder.precision : WriteParameters.DEFAULT_WRITE_PRECISION;
        this.consistency = builder.consistency;
        this.pointSettings = builder.pointSettings;
    }

    @Nonnull
    public String getUrl() {
        return this.url;
    }

    @Nullable
    public String getClientType() {
        return this.clientType;
    }

    @Nonnull
    public OkHttpClient.Builder getOkHttpClient() {
        return this.okHttpClient;
    }

    @Nonnull
    public LogLevel getLogLevel() {
        return this.logLevel;
    }

    @Nullable
    public AuthScheme getAuthScheme() {
        return this.authScheme;
    }

    @Nullable
    public char[] getToken() {
        return this.token;
    }

    @Nullable
    public String getUsername() {
        return this.username;
    }

    @Nullable
    public char[] getPassword() {
        return this.password;
    }

    @Nullable
    public String getOrg() {
        return this.org;
    }

    @Nullable
    public String getBucket() {
        return this.bucket;
    }

    @Nonnull
    public WritePrecision getPrecision() {
        return this.precision;
    }

    @Nullable
    public WriteConsistency getConsistency() {
        return this.consistency;
    }

    @Nonnull
    public PointSettings getPointSettings() {
        return this.pointSettings;
    }

    @Nonnull
    public static Builder builder() {
        return new Builder();
    }

    @NotThreadSafe
    public static class Builder {
        private String url;
        private String clientType;
        private OkHttpClient.Builder okHttpClient;
        private LogLevel logLevel;
        private AuthScheme authScheme;
        private char[] token;
        private String username;
        private char[] password;
        private String org;
        private String bucket;
        private WritePrecision precision;
        private WriteConsistency consistency;
        private final PointSettings pointSettings = new PointSettings();

        @Nonnull
        public Builder url(@Nonnull String url) {
            Arguments.checkNonEmpty(url, "url");
            this.connectionString(url);
            return this;
        }

        @Nonnull
        public Builder clientType(@Nonnull String clientType) {
            Arguments.checkNonEmpty(clientType, "clientType");
            this.clientType = clientType;
            return this;
        }

        @Nonnull
        public Builder okHttpClient(@Nonnull OkHttpClient.Builder okHttpClient) {
            Arguments.checkNotNull(okHttpClient, "OkHttpClient.Builder");
            this.okHttpClient = okHttpClient;
            return this;
        }

        @Nonnull
        public Builder logLevel(@Nonnull LogLevel logLevel) {
            Arguments.checkNotNull((Object)logLevel, "logLevel");
            this.logLevel = logLevel;
            return this;
        }

        @Nonnull
        public Builder authenticate(@Nonnull String username, @Nonnull char[] password) {
            Arguments.checkNonEmpty(username, "username");
            Arguments.checkNotNull(password, "password");
            this.authScheme = AuthScheme.SESSION;
            this.username = username;
            this.password = password;
            return this;
        }

        @Nonnull
        public Builder authenticateToken(char[] token) {
            Arguments.checkNotNull(token, "token");
            this.authScheme = AuthScheme.TOKEN;
            this.token = token;
            return this;
        }

        @Nonnull
        public Builder org(@Nullable String org) {
            this.org = org;
            return this;
        }

        @Nonnull
        public Builder bucket(@Nullable String bucket) {
            this.bucket = bucket;
            return this;
        }

        @Nonnull
        public Builder precision(@Nullable WritePrecision precision) {
            this.precision = precision;
            return this;
        }

        @Nonnull
        public Builder consistency(@Nullable WriteConsistency consistency) {
            this.consistency = consistency;
            return this;
        }

        @Nonnull
        public Builder addDefaultTag(@Nonnull String key, @Nullable String expression) {
            Arguments.checkNotNull(key, "tagName");
            this.pointSettings.addDefaultTag(key, expression);
            return this;
        }

        @Nonnull
        public Builder connectionString(@Nonnull String connectionString) {
            Arguments.checkNonEmpty(connectionString, "url");
            ParsedUrl parsedUrl = new ParsedUrl(connectionString);
            HttpUrl parse = parsedUrl.httpUrl;
            String org = parse.queryParameter("org");
            String bucket = parse.queryParameter("bucket");
            String token = parse.queryParameter("token");
            String logLevel = parse.queryParameter("logLevel");
            String readTimeout = parse.queryParameter("readTimeout");
            String writeTimeout = parse.queryParameter("writeTimeout");
            String connectTimeout = parse.queryParameter("connectTimeout");
            String precision = parse.queryParameter("precision");
            String consistency = parse.queryParameter("consistency");
            String clientType = parse.queryParameter("clientType");
            String url = parsedUrl.urlWithoutParams;
            return this.configure(url, org, bucket, token, logLevel, readTimeout, writeTimeout, connectTimeout, precision, consistency, clientType);
        }

        @Nonnull
        public Builder loadProperties() {
            Builder builder;
            block8: {
                InputStream inputStream2 = this.getClass().getResourceAsStream("/influx2.properties");
                try {
                    Properties properties = new Properties();
                    properties.load(inputStream2);
                    String url = properties.getProperty("influx2.url");
                    String org = properties.getProperty("influx2.org");
                    String bucket = properties.getProperty("influx2.bucket");
                    String token = properties.getProperty("influx2.token");
                    String logLevel = properties.getProperty("influx2.logLevel");
                    String readTimeout = properties.getProperty("influx2.readTimeout");
                    String writeTimeout = properties.getProperty("influx2.writeTimeout");
                    String connectTimeout = properties.getProperty("influx2.connectTimeout");
                    String precision = properties.getProperty("influx2.precision");
                    String consistency = properties.getProperty("influx2.consistency");
                    String clientType = properties.getProperty("influx2.clientType");
                    properties.stringPropertyNames().forEach(key -> {
                        Matcher matcher = TAGS_PROPERTY.matcher((CharSequence)key);
                        if (matcher.matches()) {
                            String tagKey = matcher.group(2);
                            this.addDefaultTag(tagKey, properties.getProperty((String)key).trim());
                        }
                    });
                    builder = this.configure(url, org, bucket, token, logLevel, readTimeout, writeTimeout, connectTimeout, precision, consistency, clientType);
                    if (inputStream2 == null) break block8;
                }
                catch (Throwable throwable) {
                    try {
                        if (inputStream2 != null) {
                            try {
                                inputStream2.close();
                            }
                            catch (Throwable throwable2) {
                                throwable.addSuppressed(throwable2);
                            }
                        }
                        throw throwable;
                    }
                    catch (IOException e) {
                        throw new IllegalStateException(e);
                    }
                }
                inputStream2.close();
            }
            return builder;
        }

        @Nonnull
        public InfluxDBClientOptions build() {
            if (this.url == null) {
                throw new IllegalStateException("The url to connect to InfluxDB has to be defined.");
            }
            if (this.okHttpClient == null) {
                this.okHttpClient = new OkHttpClient.Builder().protocols(Collections.singletonList(Protocol.HTTP_1_1));
            }
            if (this.logLevel == null) {
                this.logLevel = LogLevel.NONE;
            }
            return new InfluxDBClientOptions(this);
        }

        @Nonnull
        private Builder configure(@Nonnull String url, @Nullable String org, @Nullable String bucket, @Nullable String token, @Nullable String logLevel, @Nullable String readTimeout, @Nullable String writeTimeout, @Nullable String connectTimeout, @Nullable String precision, @Nullable String consistency, @Nullable String clientType) {
            this.url = new ParsedUrl((String)url).urlWithoutParams;
            if (org != null) {
                this.org(org);
            }
            if (bucket != null) {
                this.bucket(bucket);
            }
            if (token != null) {
                this.authenticateToken(token.toCharArray());
            }
            if (logLevel != null) {
                this.logLevel(Enum.valueOf(LogLevel.class, logLevel));
            }
            if (precision != null) {
                this.precision(Enum.valueOf(WritePrecision.class, precision));
            }
            if (consistency != null) {
                this.consistency(Enum.valueOf(WriteConsistency.class, consistency));
            }
            if (this.okHttpClient == null) {
                this.okHttpClient = new OkHttpClient.Builder().protocols(Collections.singletonList(Protocol.HTTP_1_1));
            }
            if (readTimeout != null) {
                this.okHttpClient.readTimeout(this.toDuration(readTimeout));
            }
            if (writeTimeout != null) {
                this.okHttpClient.writeTimeout(this.toDuration(writeTimeout));
            }
            if (connectTimeout != null) {
                this.okHttpClient.connectTimeout(this.toDuration(connectTimeout));
            }
            if (clientType != null) {
                this.clientType(clientType);
            }
            return this;
        }

        @Nonnull
        private Duration toDuration(@Nonnull String value) {
            Matcher matcher = DURATION_PATTERN.matcher(value);
            if (!matcher.matches()) {
                throw new InfluxException("'" + value + "' is not a valid duration");
            }
            String amount = matcher.group(1);
            String unit = matcher.group(2);
            return Duration.of(Long.parseLong(amount), switch (unit != null && !unit.isEmpty() ? unit.toLowerCase() : "ms") {
                case "ms" -> ChronoUnit.MILLIS;
                case "s" -> ChronoUnit.SECONDS;
                case "m" -> ChronoUnit.MINUTES;
                default -> throw new InfluxException("unknown unit for '" + value + "'");
            });
        }

        private static final class ParsedUrl {
            @Nonnull
            private final String urlWithoutParams;
            @Nonnull
            private final HttpUrl httpUrl;

            private ParsedUrl(@Nonnull String connectionString) {
                HttpUrl parse = HttpUrl.parse(connectionString);
                if (parse == null) {
                    throw new InfluxException("Unable to parse connection string " + connectionString);
                }
                this.httpUrl = parse;
                HttpUrl url = this.httpUrl.newBuilder().build();
                Object host = url.host().contains(":") ? "[" + url.host() + "]" : url.host();
                String urlWithoutParams = url.scheme() + "://" + (String)host + ":" + url.port() + url.encodedPath();
                this.urlWithoutParams = urlWithoutParams.endsWith("/") ? urlWithoutParams : urlWithoutParams + "/";
            }
        }
    }

    public static enum AuthScheme {
        SESSION,
        TOKEN;

    }
}

