#!/usr/bin/env python

from __future__ import print_function # Compatibility with Jython
import os
import subprocess
import sys

help = """
Start script for the QF-Test example WebSwing Server.
usage: WebSwingServer.py [--java path/to/java] [--version 24] [--executable path/to/server.jar]
    
optional arguments:
  --help, -h    Show this message
  --java        The full path to the java binary with which to launch the WebSwing server (must not contain spaces)
  --version     The major version of the WebSwing server to launch (22 and up, default: 24)
  --executable  The name of the .war or .jar file to launch, ignored on versions >= 23 (default: webswing-server.war)
"""

def parse_args():
  raw_args = sys.argv[1:]
  args = {}
  for idx, arg in enumerate(raw_args):
      if arg in ["-h", "--help"]:
        print(help)
        sys.exit(0)
      if idx % 2:
         if arg.startswith("-"):
            raise Exception("Unexpected argument value: " + arg)
      else:
        if arg not in ["--java", "--version", "--executable"]:
          raise Exception("Unexpected argument name: " + arg)
        try:
          args[arg[2:]] = raw_args[idx+1]
        except:
          pass
  return args

def is_macos():
  if "darwin" in sys.platform.lower():
    return True
  if "java" in sys.platform.lower(): # Jython
    from java.lang import System
    if "mac" in System.getProperty("os.name").lower():
      return True
  return False

try:
  args = parse_args()

  java = args.get("java")
  if not java:
    java = os.getenv('WEBSWING_JAVA')
  if not java:
    java = os.getenv('QFTEST_JAVA')
  if not java:
    raise Exception("No Java binary found. Please run via the QF-Test WebSwing example test suite.")
  print("Using Java binary at '"+java+"'")

  version = args.get("version")
  if not version:
    version = "24"

  executable = args.get("executable")
  if not executable:
    executable = "webswing-server.war"
  if int(version) >= 23:
      executable = "server/webswing-jetty-launcher.jar"

  webswing_args = [
        "-Dwebswing.disableAutoStartBrowser=true",
        "-jar", executable,
        "-j", "jetty.properties"]
  if int(version) >= 23:
    webswing_args += [
        "-serveradmin",
        "-pfa", "admin/webswing-admin.properties",
        "-adminctx", "/admin",
        "-aw", "admin/webswing-admin-server.war"]
    
  if is_macos():
    # Launch Java in x86 mode since WebSwing does not support native ARM
    subprocess.check_call(['arch','-x86_64', java] + webswing_args)
  else:
    subprocess.check_call([java] + webswing_args)

except Exception as e:
  print(e)
  print(help)
  sys.exit(1)
