// {{{ copyright

/********************************************************************
 *
 * The contents of this file are subject to the Mozilla Public
 * License Version 1.1 (the "License"); you may not use this file
 * except in compliance with the License. You may obtain a copy of
 * the License at http://www.mozilla.org/MPL/
 *
 * Software distributed under the License is distributed on an "AS
 * IS" basis, WITHOUT WARRANTY OF ANY KIND, either express or
 * implied. See the License for the specific language governing
 * rights and limitations under the License.
 *
 * The Original Code is qfs.de code.
 *
 * The Initial Developer of the Original Code is Gregor Schmid.
 * Portions created by Gregor Schmid are
 * Copyright (C) 2000 Quality First Software, Gregor Schmid.
 * All Rights Reserved.
 *
 * Contributor(s):
 *
 *******************************************************************/

// }}}

package de.qfs.lib.log;

// {{{ doc

/**
 * This class provides a default implementation of the {@link LogUser LogUser}
 * interface that reduces the effort of writing a new {@link LogFilter
 * LogFilter} to implementing the {@link #isUseful isUseful} method. <p>
 *
 * The {@link #filter filter} methods are implemented according to the filter
 * mode setting as documented in {@link LogUser LogUser}. <p>
 *
 * Useful messages are passed to a {@link LogWriter LogWriter} that can be set
 * via {@link #setLogWriter setLogWriter}.
 *
 * @author      Gregor Schmid
 * @since       0.98.0
 */

// }}}
public abstract class AbstractLogUser
    implements LogUser
{
    // {{{ variables

    /**
     * The {@link LogUser LogUser} mode.
     */
    protected int mode = FILTER_NONE;

    /**
     * The LogWriter to write entries to.
     */
    private LogWriter writer;

    // }}}

    //----------------------------------------------------------------------
    // Constructor
    //----------------------------------------------------------------------
    // {{{ AbstractLogUser(int,LogWriter)

    /**
     * Create a new AbstractLogUser.
     *
     * @param   mode    The {@link LogUser LogUser} mode.
     * @param   writer  The LogWriter to write entries to.
     */
    public AbstractLogUser(int mode, LogWriter writer)
    {
        this.mode = mode;
        this.writer = writer;
    }

    // }}}

    //----------------------------------------------------------------------
    // Public methods.
    //----------------------------------------------------------------------
    // {{{ getLogWriter

    /**
     * Get the LogWriter of the AbstractLogUser.
     *
     * @return  The writer of the AbstractLogUser.
     */
    public final LogWriter getLogWriter()
    {
        return writer;
    }

    // }}}
    // {{{ setLogWriter

    /**
     * Set the LogWriter of the AbstractLogUser.
     *
     * @param   writer  The writer to set.
     */
    public final void setLogWriter(LogWriter writer)
    {
        this.writer = writer;
    }

    // }}}

    //----------------------------------------------------------------------
    // The LogUser interface.
    //----------------------------------------------------------------------
    // {{{ filter(LogEntry)

    /**
     * Filter a LogEntry and possibly write it to the LogWriter.
     *
     * @param   entry        The entry to filter.
     *
     * @return  True if the entry should be passed on to the next filter.
     */
    @Override
    public boolean filter (LogEntry entry)
    {
        boolean used = false;
        if (isUseful(entry)) {
            if (writer != null) {
                writer.write(entry);
            }
            used = true;
        }
        return mode != FILTER_ALL
            && ((used && mode != FILTER_USED)
                || (!used && mode != FILTER_UNUSED));
    }

    // }}}
    // {{{ filter(LogEntry[])

    /**
     * Filter an array of LogEntries, possibly writing some of them to the
     * LogWriter.
     *
     * @param   entries The entries to filter.
     */
    @Override
    public LogEntry[] filter (LogEntry[] entries)
    {
        // Its probably a lot faster to count first than to use Vectors and
        // copy later

        int use = 0;

        for (int i = 0; i < entries.length; i++) {
            if (isUseful(entries[i])) {
                use++;
            }
        }

        // There are two possible shorcuts: nothing filtered ...
        if (use == entries.length) {
            if (writer != null) {
                writer.write(entries);
            }
            if (mode == FILTER_ALL || mode == FILTER_USED) {
                return null;
            }
            return entries;
        }

        // ... and all filtered
        if (use == 0) {
            if (mode == FILTER_ALL || mode == FILTER_UNUSED) {
                return null;
            }
            return entries;
        }

        // we really have to split the array
        LogEntry[] unused = (mode == FILTER_USED)
            ? new LogEntry[entries.length - use] : null;
        LogEntry[] used = null;
        if (writer != null || mode == FILTER_UNUSED) {
            used = new LogEntry[use];
        }

        int cUsed = 0;
        int cUnused = 0;
        for (int i = 0; i < entries.length; i++) {
            if (isUseful(entries[i])) {
                if (used != null) {
                    used[cUsed++] = entries[i];
                }
            } else {
                if (unused != null) {
                    unused[cUnused++] = entries[i];
                }
            }
        }

        if (writer != null) {
            writer.write(used);
        }
        switch (mode) {
        case FILTER_ALL:
            return null;
        case FILTER_NONE:
            return entries;
        case FILTER_USED:
            return unused;
        case FILTER_UNUSED:
            return used;
        }
        // Never reached
        return null;
    }

    // }}}
    // {{{ setFilterMode

    /**
     * Configure the AbstractLogUser's filter mode.
     *
     * @param   mode    The mode to set.
     */
    @Override
    public void setFilterMode(int mode)
    {
        this.mode = mode;
    }

    // }}}

    //----------------------------------------------------------------------
    // The actual filter method
    //----------------------------------------------------------------------
    // {{{ isUseful

    /**
     * Decide whether a log message is useful and should be passed to the
     * LogWriter.
     *
     * @param   entry   The LogEntry to check.
     *
     * @return  True if the entry is considered useful, false to skip it.
     */
    protected abstract boolean isUseful(LogEntry entry);

    // }}}
}
