// {{{ copyright

/********************************************************************
 *
 * The contents of this file are subject to the Mozilla Public
 * License Version 1.1 (the "License"); you may not use this file
 * except in compliance with the License. You may obtain a copy of
 * the License at http://www.mozilla.org/MPL/
 *
 * Software distributed under the License is distributed on an "AS
 * IS" basis, WITHOUT WARRANTY OF ANY KIND, either express or
 * implied. See the License for the specific language governing
 * rights and limitations under the License.
 *
 * The Original Code is qfs.de code.
 *
 * The Initial Developer of the Original Code is Oliver Brandt.
 * Portions created by Oliver Brandt are
 * Copyright (C) 2000 ATecoM GmbH.
 * Portions created by Gregor Schmid are
 * Copyright (C) 2000 Quality First Software, Gregor Schmid.
 * All Rights Reserved.
 *
 * Contributor(s):
 * Gregor Schmid, Quality First Software
 *
 *******************************************************************/

// }}}

package de.qfs.lib.log;

// {{{ imports

import java.text.DateFormat;
import java.text.SimpleDateFormat;

import java.util.Date;

// }}}

// {{{ doc

/**
 * This is a default implementation of the {@link LogFormat LogFormat}
 * interface that produces the same result as the {@link LogEntry#toString
 * LogEntry.toString} method before Version 0.98.0
 *
 * @author      Oliver Brandt
 * @author      Gregor Schmid
 * @since       0.98.0
 */

// }}}
public class DefaultLogFormat
    implements LogFormat
{
    // {{{ variables

    /**
     * Helper to print timestamps.
     */
    private static Date date = new Date();

    /**
     * StringBuffer used for message output.
     */
    private static StringBuffer buffer = new StringBuffer ();

    /**
     * Helper to print timestamps.
     */
    private DateFormat df;

    // }}}

    //----------------------------------------------------------------------
    // Constructors
    //----------------------------------------------------------------------
    // {{{ DefaultLogFormat()

    /**
     * Create a new DefaultLogFormat that uses a default {@link
     * java.text.DateFormat DateFormat} of HH:mm:ss.SSS for timestamps.
     */
    public DefaultLogFormat()
    {
        df = new SimpleDateFormat("HH:mm:ss.SSS");
    }

    // }}}
    // {{{ DefaultLogFormat(DateFormat)

    /**
     * Create a new DefaultLogFormat.
     *
     * @param   df      The format to use for timestamps.
     */
    public DefaultLogFormat(DateFormat df)
    {
        this.df = df;
    }

    // }}}

    // {{{ format

    /**
     * Format one {@link LogEntry LogEntry}.
     *
     * @param   entry   The LogEntry to format.
     *
     * @return  A String representation for the LogEntry.
     */
    @Override
    public String format(LogEntry entry)
    {
        // synchronization also protects the buffer
        synchronized(date) {
            date.setTime(entry.getTimestamp());
            buffer.setLength(0);
            buffer.append(entry.getLevel());
            buffer.append(" (");
            buffer.append(df.format(date));
            buffer.append(") ");
            buffer.append(entry.getThread());
            buffer.append(" ");
            buffer.append(entry.getClazz());
            buffer.append(".");
            buffer.append(entry.getMethod());
            buffer.append(": ");
            buffer.append(entry.getMessage());
            return buffer.toString();
        }
    }

    // }}}
}
