// {{{ copyright

/********************************************************************
 *
 * The contents of this file are subject to the Mozilla Public
 * License Version 1.1 (the "License"); you may not use this file
 * except in compliance with the License. You may obtain a copy of
 * the License at http://www.mozilla.org/MPL/
 *
 * Software distributed under the License is distributed on an "AS
 * IS" basis, WITHOUT WARRANTY OF ANY KIND, either express or
 * implied. See the License for the specific language governing
 * rights and limitations under the License.
 *
 * The Original Code is qfs.de code.
 *
 * The Initial Developer of the Original Code is Gregor Schmid.
 * Portions created by Gregor Schmid are
 * Copyright (C) 2000 Quality First Software, Gregor Schmid.
 * All Rights Reserved.
 *
 * Contributor(s):
 *
 *******************************************************************/

// }}}

package de.qfs.lib.log;

// {{{ imports

import java.io.File;
import java.io.FileOutputStream;
import java.io.OutputStreamWriter;
import java.io.IOException;
import java.io.PrintWriter;

// }}}

// {{{ doc

/**
 * An extension of the {@link StreamLogWriter StreamLogWriter} that writes
 * {@link LogEntry LogEntries} to a {@link java.io.File File}. <p>
 *
 * The constructors leave three choices for how the file should be opened:
 *
 * <dl>
 * <dt>{@link #MODE_CREATE MODE_CREATE}</dt>
 * <dd>Create a new file, overwriting existing files (the default).</dd>
 * <dt>{@link #MODE_APPEND MODE_APPEND}</dt>
 * <dd>If the file already exists, append to it, otherwise create a new
 * file.</dd>
 * <dt>{@link #MODE_UNIQUE MODE_UNIQUE}</dt>
 * <dd>First make the filename unique by appending numbers after the basename,
 * starting with 1, until an unused name is found. Then open that file. So the
 * first time you create a FileLogWriter for the file <code>myname.log</code>,
 * this will create <code>myname1.log</code>, the next time its
 * <code>myname2.log</code> and so on. </dd>
 * </dl>
 *
 * The FileLogWriter has two modes of operation: it can simply write the logs
 * to the file or it can close the file after each write, reopening it for the
 * next write. This takes a little more time but shouldn't hurt too badly,
 * since messages typically arrive in bunches.
 *
 * @author      Gregor Schmid
 * @since       0.98.0
 */

// }}}
public class FileLogWriter
    extends StreamLogWriter
{
    // {{{ variables

    /**
     * Create a new log file, overwriting existing files.
     */
    public final static int MODE_CREATE = 0;

    /**
     * Append to a log file, creating a new one if necessary.
     */
    public final static int MODE_APPEND = 1;

    /**
     * Create a new log file. A number appended to the basename is incremented
     * until the file name is unique.
     */
    public final static int MODE_UNIQUE = 2;

    /**
     * LevelFilter for convenience methods.
     */
    private static LevelFilter lastLevelFilter;

    /**
     * The log file to write to.
     */
    protected File file;

    /**
     * Whether to close the log file after each write.
     */
    protected boolean keepClosed;

    // }}}

    //----------------------------------------------------------------------
    // Constructors
    //----------------------------------------------------------------------
    // {{{ FileLogWriter(String,String)

    /**
     * Create a new FileLogWriter that uses a {@link DefaultLogFormat
     * DefaultLogFormat} to write {@link LogEntry LogEntries} to a file.
     *
     * @param   client  Name of the client, used by qflog.
     * @param   file    The name of the file to print to.
     *
     * @throws  IOException     If the file cannot be opened for writing.
     */
    public FileLogWriter(String client, String file)
        throws IOException
    {
        openFile(client, file, MODE_CREATE);
    }

    // }}}
    // {{{ FileLogWriter(String,String,int,boolean)

    /**
     * Create a new FileLogWriter that uses a {@link DefaultLogFormat
     * DefaultLogFormat} to write {@link LogEntry LogEntries} to a file.
     *
     * @param   client  Name of the client, used by qflog.
     * @param   file    The name of the file to print to.
     * @param   mode    How the file should be created. Must be one of
     *                  {@link #MODE_CREATE MODE_CREATE}, {@link #MODE_APPEND
     *                  MODE_APPEND}, or {@link #MODE_UNIQUE MODE_UNIQUE}.
     * @param   keepClosed Whether to close the file after each write.
     *
     * @throws  IOException     If the file cannot be opened for writing.
     */
    public FileLogWriter(String client, String file, int mode,
                         boolean keepClosed)
        throws IOException
    {
        this.keepClosed = keepClosed;
        openFile(client, file, mode);
    }

    // }}}
    // {{{ FileLogWriter(String,String,LogFormat)

    /**
     * Create a new FileLogWriter that writes {@link LogEntry LogEntries} to
     * a file.
     *
     * @param   client  Name of the client, used by qflog.
     * @param   file    The name of the file to print to.
     * @param   format  The format used to print LogEntries.
     *
     * @throws  IOException     No longer thrown but kept in the interface for compatibility.
     */
    public FileLogWriter(String client, String file, LogFormat format)
        throws IOException
    {
        super (format);
        openFile(client, file, MODE_CREATE);
    }

    // }}}
    // {{{ FileLogWriter(String,String,int,boolean,LogFormat)

    /**
     * Create a new FileLogWriter that writes {@link LogEntry LogEntries} to
     * a file.
     *
     * @param   client  Name of the client, used by qflog.
     * @param   file    The name of the file to print to.
     * @param   mode    How the file should be created. Must be one of
     *                  {@link #MODE_CREATE MODE_CREATE}, {@link #MODE_APPEND
     *                  MODE_APPEND}, or {@link #MODE_UNIQUE MODE_UNIQUE}.
     * @param   keepClosed Whether to close the file after each write.
     * @param   format  The format used to print LogEntries.
     *
     * @throws  IOException     No longer thrown but kept in the interface for compatibility.
     */
    public FileLogWriter(String client, String file, int mode,
                         boolean keepClosed, LogFormat format)
        throws IOException
    {
        super (format);
        this.keepClosed = keepClosed;
        openFile(client, file, mode);
    }

    // }}}

    //----------------------------------------------------------------------
    // Static convenience methods.
    //----------------------------------------------------------------------
    // {{{ logToFile(String,String,int,boolean)

    /**
     * Log messages to a log file by creating a LevelFilter with a
     * FileLogWriter and adding it to the Log filter chain. The log file
     * will be in the format recognized by the qflog log server.
     *
     * @param   client  Name of the client, used by qflog.
     * @param   file    The name of the file to save in.
     * @param   mode    How the file should be created. Must be one of
     *                  {@link #MODE_CREATE MODE_CREATE}, {@link #MODE_APPEND
     *                  MODE_APPEND}, or {@link #MODE_UNIQUE MODE_UNIQUE}.
     * @param   keepClosed Whether to close the file after each write.
     *
     * @return The new LevelFilter. Unless you change its level via {@link
     * de.qfs.lib.log.LevelFilter#setLevel LevelFilter.setLevel}, it will log
     * all messages to the file. If you only call this method once in your
     * application, you can use {@link #stopLogging stopLogging} to remove the
     * filter and close the writer, otherwise you have to save this reference
     * to the filter and clean up yourself.
     *
     * @throws  IOException     No longer thrown but kept in the interface for compatibility.
     */
    public static LevelFilter logToFile(String client, String file, int mode,
                                        boolean keepClosed)
        throws IOException
    {
        lastLevelFilter = new LevelFilter
            (-1, new FileLogWriter (client, file, mode, keepClosed));
        Log.addFilter(lastLevelFilter);
        return lastLevelFilter;
    }

    // }}}
    // {{{ logToFile(String,String,int,boolean,LogFormat)

    /**
     * Log messages to a log file by creating a LevelFilter with a
     * FileLogWriter and adding it to the Log filter chain. The log file
     * will be in the format recognized by the qflog log server.
     *
     * @param   client  Name of the client, used by qflog.
     * @param   file    The name of the file to save in.
     * @param   mode    How the file should be created. Must be one of
     *                  {@link #MODE_CREATE MODE_CREATE}, {@link #MODE_APPEND
     *                  MODE_APPEND}, or {@link #MODE_UNIQUE MODE_UNIQUE}.
     * @param   keepClosed Whether to close the file after each write.
     * @param   format  The format used to print LogEntries.
     *
     * @return The new LevelFilter. Unless you change its level via {@link
     * de.qfs.lib.log.LevelFilter#setLevel LevelFilter.setLevel}, it will log
     * all messages to the file. If you only call this method once in your
     * application, you can use {@link #stopLogging stopLogging} to remove the
     * filter and close the writer, otherwise you have to save this reference
     * to the filter and clean up yourself.
     *
     * @throws  IOException     No longer thrown but kept in the interface for compatibility.
     */
    public static LevelFilter logToFile(String client, String file, int mode,
                                        boolean keepClosed, LogFormat format)
        throws IOException
    {
        lastLevelFilter = new LevelFilter
            (-1, new FileLogWriter (client, file, mode, keepClosed, format));
        Log.addFilter(lastLevelFilter);
        return lastLevelFilter;
    }

    // }}}
    // {{{ stopLogging

    /**
     * Remove the last {@link LevelFilter LevelFilter} instance created with
     * {@link #logToFile logToFile} from the filter chain and close the log
     * file. Use this method if you only call {@link #logToFile logToFile}
     * once in your application.
     */
    public static void stopLogging()
    {
        if (lastLevelFilter != null) {
            lastLevelFilter.getLogWriter().close();
            Log.removeFilter(lastLevelFilter);
            lastLevelFilter = null;
        }
    }

    // }}}

    //----------------------------------------------------------------------
    // The LogWriter interface
    //----------------------------------------------------------------------
    // {{{ write(LogEntry)

    /**
     * Write one LogEntry.
     *
     * @param   entry   The entry to write.
     */
    @Override
    public synchronized void write(LogEntry entry)
    {
        if (closed) {
            return;
        }
        try {
            if (keepClosed) {
                pw = new PrintWriter
                    (new OutputStreamWriter (new FileOutputStream (file.getPath(), true), "UTF-8"));
            }
            super.write(entry);
            if (keepClosed) {
                pw.close();
                pw = null;
            }
        } catch (IOException ex) {
            // In this case we simply lose
        }
    }

    // }}}
    // {{{ write(LogEntry[])

    /**
     * Write an array of LogEntires in one go. Clients of the FileLogWriter
     * should use this method in preference to {@link #write(LogEntry)
     * write(LogEntry)}, since it is more efficient.
     *
     * @param   entries The entries to write.
     */
    @Override
    public synchronized void write(LogEntry[] entries)
    {
        if (closed) {
            return;
        }
        try {
            if (keepClosed) {
                pw = new PrintWriter
                    (new OutputStreamWriter (new FileOutputStream (file.getPath(), true), "UTF-8"));
            }
            super.write(entries);
            if (keepClosed) {
                pw.close();
                pw = null;
            }
        } catch (IOException ex) {
            // In this case we simply lose
        }
    }

    // }}}

    //----------------------------------------------------------------------
    // Helper methods
    //----------------------------------------------------------------------
    // {{{ openFile

    /**
     * Open the file for writing.
     *
     * @param   client  Name of the client, used by qflog.
     * @param   mode    How the file should be created. Must be one of
     *                  {@link #MODE_CREATE MODE_CREATE}, {@link #MODE_APPEND
     *                  MODE_APPEND}, or {@link #MODE_UNIQUE MODE_UNIQUE}.
     */
    protected void openFile(String client, String filename, int mode)
    {
        boolean printClient = mode != MODE_APPEND;
        file = new File (filename);
        File pFile = new File (file.getParent());
        if (! pFile.isDirectory()) {
            // Create directory on the fly - best effort
            pFile.mkdirs();
        }
        try {
            switch(mode) {
            case MODE_CREATE:
                pw = new PrintWriter (new OutputStreamWriter (new FileOutputStream (file), "UTF-8"));
                break;
            case MODE_APPEND:
                printClient = ! file.exists();
                pw = new PrintWriter (new OutputStreamWriter (new FileOutputStream (filename, true), "UTF-8"));
                break;
            case MODE_UNIQUE:
                String suffix = "";
                int pos = filename.lastIndexOf('.');
                if (pos > 0) {
                    suffix = filename.substring(pos);
                    filename = filename.substring(0, pos);
                }
                for (int id = 1; true; id++) {
                    file = new File (filename + id + suffix);
                    if (! file.exists()) {
                        pw = new PrintWriter (new OutputStreamWriter (new FileOutputStream (file), "UTF-8"));
                        break;
                    }
                }
                break;
            default:
                throw new IllegalArgumentException
                    ("Argument mode must be one of MODE_CREATE, " +
                     "MODE_APPEND or MODE_UNIQUE.");
            }

            // if we get here, file is set correctly and can be written to

            if (printClient) {
                pw.println("client name: " + client);
                pw.flush();
            }

            if (keepClosed) {
                pw.close();
                pw = null;
            } else {
                mustClose = true;
            }
        } catch (IOException ex) {
            closed = true;
        }
    }

    // }}}
}
