// {{{ copyright

/********************************************************************
 *
 * The contents of this file are subject to the Mozilla Public
 * License Version 1.1 (the "License"); you may not use this file
 * except in compliance with the License. You may obtain a copy of
 * the License at http://www.mozilla.org/MPL/
 *
 * Software distributed under the License is distributed on an "AS
 * IS" basis, WITHOUT WARRANTY OF ANY KIND, either express or
 * implied. See the License for the specific language governing
 * rights and limitations under the License.
 *
 * The Original Code is qfs.de code.
 *
 * The Initial Developer of the Original Code is Gregor Schmid.
 * Portions created by Gregor Schmid are
 * Copyright (C) 1999 Quality First Software, Gregor Schmid.
 * All Rights Reserved.
 *
 * Contributor(s):
 *
 *******************************************************************/

// }}}

package de.qfs.lib.log;

// {{{ imports




// }}}

/**
 * Helper class for buffering log messages that may be needed again, if
 * an error message triggers the flushing of logs. <p>
 *
 * For efficiency reasons there is one array per message part, i.e. one array
 * for the levels of all buffered messages, one array for the Threads etc.
 *
 * @author      Gregor Schmid
 */
class FlushBuffer
{
    // {{{ variables

    /**
     * The buffers size.
     */
    private int size;

    /**
     * The levels of the messages.
     */
    private int[] levels;

    /**
     * The timestamps of the messages.
     */
    private long[] times;

    /**
     * The Threads of the messages.
     */
    private Thread[] threads;

    /**
     * The Classes that logged the messages.
     */
    private String[] classes;

    /**
     * The methods that logged the messages.
     */
    private String[] methods;

    /**
     * The actual messages.
     */
    private String[] messages;

    /**
     * The current head mark.
     */
    private int head = 0;

    /**
     * The current tail mark.
     */
    private int tail = 0;

    // }}}

    // {{{ constructor

    /**
     * Create a new FlushBuffer.
     *
     * @param   fbsize  The size of the buffer.
     */
    FlushBuffer(int fbsize)
    {
        if (fbsize < 0) {
            throw new IllegalArgumentException("Illegal size " + fbsize +
                                               ". Must be >= 0");
        }

        // one extra for the empty cell head points to
        size = fbsize + 1;
        levels = new int[size];
        times = new long[size];
        threads = new Thread[size];
        classes = new String[size];
        methods = new String[size];
        messages = new String[size];
    }

    // }}}

    // {{{ buffer

    /**
     * Store one message in the buffer.
     *
     * @param   level   The log level of the message.
     * @param   timestamp The time of the message in milliseconds.
     * @param   clazz   The class of the sender of the message.
     * @param   method  The method that sent the message.
     * @param   message The message itself.
     */
    void buffer(int level, long timestamp,
                String clazz, String method, String message)
    {
        Thread thread = Thread.currentThread();

        int pos;
        // I don't see a way around synchronization here
        synchronized (this) {
            pos = head++;
            if (head == size) {
                head = 0;
            }
            if (head == tail) {
                tail++;
                if (tail == size) {
                    tail = 0;
                }
            }
        }
        levels[pos] = level;
        times[pos] = timestamp;
        threads[pos] = thread;
        classes[pos] = clazz;
        methods[pos] = method;
        messages[pos] = message;
    }

    // }}}
    // {{{ flush

    /**
     * Flush the buffer.
     *
     * @return  An array of the messages stored in the buffer.
     */
    synchronized LogEntry[] flush()
    {
        int count = count();

        LogEntry[] ret = new LogEntry[count];
        for (int i = 0; i < count; i++, tail++) {
            if (tail == size) {
                tail = 0;
            }
            ret[i] = new LogEntry
                (levels[tail], times[tail], threads[tail].getName(),
                 classes[tail], methods[tail], messages[tail]);
        }
        return ret;
    }

    // }}}
    // {{{ getSize

    /**
     * Get the size of the FlushBuffer.
     *
     * @return  The size of the FlushBuffer.
     */
    public int getSize()
    {
        return size - 1;
    }

    // }}}
    // {{{ resize

    /**
     * Resize the FlushBuffer.
     *
     * @param   newSize The new size of the buffer.
     */
    void resize(int newSize)
    {
        if (newSize == size - 1) {
            return;
        }

        int[] newLevels = new int[newSize + 1];
        long[] newTimes = new long[newSize + 1];
        Thread[] newThreads = new Thread[newSize + 1];
        String[] newClasses = new String[newSize + 1];
        String[] newMethods = new String[newSize + 1];
        String[] newMessages = new String[newSize + 1];

        int count = count();

        if (count > newSize) {
            if (head > tail) {
                System.arraycopy(levels, tail + (count - newSize),
                                 newLevels, 0, newSize);
                System.arraycopy(times, tail + (count - newSize),
                                 newTimes, 0, newSize);
                System.arraycopy(threads, tail + (count - newSize),
                                 newThreads, 0, newSize);
                System.arraycopy(classes, tail + (count - newSize),
                                 newClasses, 0, newSize);
                System.arraycopy(methods, tail + (count - newSize),
                                 newMethods, 0, newSize);
                System.arraycopy(messages, tail + (count - newSize),
                                 newMessages, 0, newSize);
            } else if (head >= newSize) {
                System.arraycopy(levels, head - newSize,
                                 newLevels, 0, newSize);
                System.arraycopy(times, head - newSize,
                                 newTimes, 0, newSize);
                System.arraycopy(threads, head - newSize,
                                 newThreads, 0, newSize);
                System.arraycopy(classes, head - newSize,
                                 newClasses, 0, newSize);
                System.arraycopy(methods, head - newSize,
                                 newMethods, 0, newSize);
                System.arraycopy(messages, head - newSize,
                                 newMessages, 0, newSize);
            } else {
                int part1 = newSize - head;
                System.arraycopy(levels, size - part1,
                                 newLevels, 0, part1);
                System.arraycopy(times, size - part1,
                                 newTimes, 0, part1);
                System.arraycopy(threads, size - part1,
                                 newThreads, 0, part1);
                System.arraycopy(classes, size - part1,
                                 newClasses, 0, part1);
                System.arraycopy(methods, size - part1,
                                 newMethods, 0, part1);
                System.arraycopy(messages, size - part1,
                                 newMessages, 0, part1);
                if (head > 0) {
                    System.arraycopy(levels, 0,
                                     newLevels, part1, head);
                    System.arraycopy(times, 0,
                                     newTimes, part1, head);
                    System.arraycopy(threads, 0,
                                     newThreads, part1, head);
                    System.arraycopy(classes, 0,
                                     newClasses, part1, head);
                    System.arraycopy(methods, 0,
                                     newMethods, part1, head);
                    System.arraycopy(messages, 0,
                                     newMessages, part1, head);
                }
            }
        } else if (count > 0) {
            if (head > tail) {
                System.arraycopy(levels, tail,
                                 newLevels, 0, count);
                System.arraycopy(times, tail,
                                 newTimes, 0, count);
                System.arraycopy(threads, tail,
                                 newThreads, 0, count);
                System.arraycopy(classes, tail,
                                 newClasses, 0, count);
                System.arraycopy(methods, tail,
                                 newMethods, 0, count);
                System.arraycopy(messages, tail,
                                 newMessages, 0, count);
            } else {
                System.arraycopy(levels, tail,
                                 newLevels, 0, size - tail);
                System.arraycopy(times, tail,
                                 newTimes, 0, size - tail);
                System.arraycopy(threads, tail,
                                 newThreads, 0, size - tail);
                System.arraycopy(classes, tail,
                                 newClasses, 0, size - tail);
                System.arraycopy(methods, tail,
                                 newMethods, 0, size - tail);
                System.arraycopy(messages, tail,
                                 newMessages, 0, size - tail);
                if (head > 0) {
                    System.arraycopy(levels, 0,
                                     newLevels, size - tail, head);
                    System.arraycopy(times, 0,
                                     newTimes, size - tail, head);
                    System.arraycopy(threads, 0,
                                     newThreads, size - tail, head);
                    System.arraycopy(classes, 0,
                                     newClasses, size - tail, head);
                    System.arraycopy(methods, 0,
                                     newMethods, size - tail, head);
                    System.arraycopy(messages, 0,
                                     newMessages, size - tail, head);
                }
            }
        }

        levels = newLevels;
        times = newTimes;
        threads = newThreads;
        classes = newClasses;
        methods = newMethods;
        messages = newMessages;

        size = newSize + 1;
        tail = 0;
        head = Math.min(count, newSize);
    }

    // }}}

    //----------------------------------------------------------------------
    // Helper methods
    //----------------------------------------------------------------------
    // {{{ count

    /**
     * Get the number of entries in the buffer.
     *
     * @return  The number of entries in the buffer.
     */
    private int count()
    {
        return head >= tail ? head - tail : head - tail + size;
    }

    // }}}

}
