// {{{ copyright

/********************************************************************
 *
 * The contents of this file are subject to the Mozilla Public
 * License Version 1.1 (the "License"); you may not use this file
 * except in compliance with the License. You may obtain a copy of
 * the License at http://www.mozilla.org/MPL/
 *
 * Software distributed under the License is distributed on an "AS
 * IS" basis, WITHOUT WARRANTY OF ANY KIND, either express or
 * implied. See the License for the specific language governing
 * rights and limitations under the License.
 *
 * The Original Code is qfs.de code.
 *
 * The Initial Developer of the Original Code is Gregor Schmid.
 * Portions created by Gregor Schmid are
 * Copyright (C) 1999 Quality First Software, Gregor Schmid.
 * All Rights Reserved.
 *
 * Contributor(s):
 *
 *******************************************************************/

// }}}

package de.qfs.lib.log;

// {{{ imports

import java.lang.reflect.Array;

import java.util.Collection;
import java.util.ConcurrentModificationException;
import java.util.List;
import java.util.Map;
import java.util.Map.Entry;
import java.util.Queue;
import java.util.Set;
import java.util.concurrent.ConcurrentLinkedQueue;

import javax.annotation.Nullable;

import lombok.NonNull;

// }}}
// {{{ class GenericLevelAwareBuilder

/**
* Helper class for building log messages.
*
*/
@SuppressWarnings("unchecked") // Smell to avoid the getThis() pattern
public class GenericLevelAwareLogBuilder<T extends GenericLevelAwareLogBuilder<T>> {
    // {{{ constants

    /**
     * Typical length of interesting elements in an array.
     */
    private final static int DEFAULT_NUMBER_OF_ELEMENTS_TO_LOG_IN_ARRAY = 30;

    /**
     * Typical length of interesting elements in a map.
     */
    private final static int DEFAULT_NUMBER_OF_ELEMENTS_TO_LOG_IN_MAP = 50;

    /**
     * Educated guess for average message sized, used to initialize StringBuilders for
     * constructing messages.
     */
    private final static int AVERAGE_MESSAGE_SIZE = 200;
    // }}}

    // {{{ variables

    /**
     * The buffer used for building the string.
     */
    protected final StringBuilder sb = new StringBuilder(AVERAGE_MESSAGE_SIZE);

    /**
     * The method for which the message is built.
     */
    protected final String method;

    protected int numberOfArrayElementsToLog = DEFAULT_NUMBER_OF_ELEMENTS_TO_LOG_IN_ARRAY;

    protected int numberOfMapElementsToLog = DEFAULT_NUMBER_OF_ELEMENTS_TO_LOG_IN_MAP;

    protected final QFLoggerInterface logger;


    /**
     * The level for which the Builder is used.
     */
    protected final int level;

    /**
     * Flag to indicate if details should be included
     */
    protected final boolean useDetails;

    /**
     * The referenced source line
     */
    protected int lineNumber;

    protected Queue<LogEntry> triggeredLogEntries = null;

    private final Object addAllMonitor = new Object();

    @Nullable
    protected final List<LogObjectDumper> objectDumpers;

    // }}}

    // --------------------------------------------------------------------------------------
    // Constructors
    // --------------------------------------------------------------------------------------
    // {{{ GenericLevelAwareLogBuilder

    /**
     * Create a new GenericLevelAwareLogBuilder.
     *
     * @param level
     *            The level for which the Builder is used.
     * @param method
     *            The method to log for.
     */
    protected GenericLevelAwareLogBuilder(final QFLoggerInterface logger, final int level,
                                          final String method, final int lineNumber,
                                          @Nullable final List<LogObjectDumper> objectDumpers)
    {
        this.logger = logger;
        this.level = level;
        this.method = method;
        this.lineNumber = lineNumber;
        this.objectDumpers = objectDumpers;

        // If the enclosing Logger's level is at least one higher as the set level, details
        // should be included in the message.
        this.useDetails = (logger.getLevel() >= level + 1);
    }

    // }}}
    // {{{ setNumberOfArrayElementsToLog(int)

    public T setNumberOfArrayElementsToLog(final int numberOfArrayElementsToLog) {
        synchronized(addAllMonitor) {
            this.numberOfArrayElementsToLog = numberOfArrayElementsToLog;
        }
        return (T) this;
    }

    // }}}
    // {{{ setNumberOfMapElementsToLog(int)

    public T setNumberOfMapElementsToLog(final int numberOfMapElementsToLog) {
        synchronized(addAllMonitor) {
            this.numberOfMapElementsToLog = numberOfMapElementsToLog;
        }
        return (T) this;
    }

    // }}}
    // {{{ add(boolean)

    /**
     * Add a boolean to the log message.
     *
     * @param       val       The boolean to add.
     *
     * @return      The builder itself for chaining.
     */
    public T add(final boolean val)
    {
        sb.append(val);
        return (T) this;
    }

    // }}}
    // {{{ add(byte)

    /**
     * Add a byte to the log message.
     *
     * @param       val       The byte to add.
     *
     * @return      The builder itself for chaining.
     */
    public T add(final byte val)
    {
        sb.append(val);
        return (T) this;
    }

    // }}}
    // {{{ add(char)

    /**
     * Add a char to the log message.
     *
     * @param       val       The char to add.
     *
     * @return      The builder itself for chaining.
     */
    public T add(final char val)
    {
        sb.append(val);
        return (T) this;
    }

    // }}}
    // {{{ add(int)

    /**
     * Add an int to the log message.
     *
     * @param       val       The int to add.
     *
     * @return      The builder itself for chaining.
     */
    public T add(final int val)
    {
        sb.append(val);
        return (T) this;
    }

    // }}}
    // {{{ addHex(int)

    /**
     * Add an int in hexadecimal notation to the log message.
     *
     * @param       val       The int to add.
     *
     * @return      The builder itself for chaining.
     */
    public T addHex(final int val)
    {
        sb.append("0x");
        sb.append(Integer.toHexString(val));
        return (T) this;
    }

    // }}}
    // {{{ addHex(Integer)

    /**
     * Add an Integer in hexadecimal notation to the log message.
     *
     * @param       val       The int to add.
     *
     * @return      The builder itself for chaining.
     */
    public T addHex(final Integer val)
    {
        if (val != null) {
            return addHex(val.intValue());
        }
        sb.append("null");
        return (T) this;
    }

    // }}}
    // {{{ add(long)

    /**
     * Add a long to the log message.
     *
     * @param       val       The long to add.
     *
     * @return      The builder itself for chaining.
     */
    public T add(final long val)
    {
        sb.append(val);
        return (T) this;
    }

    // }}}
    // {{{ addHex(long)

    /**
     * Add an long in hexadecimal notation to the log message.
     *
     * @param       val       The long to add.
     *
     * @return      The builder itself for chaining.
     */
    public T addHex(final long val)
    {
        sb.append("0x");
        sb.append(Long.toHexString(val));
        return (T) this;
    }

    // }}}
    // {{{ addHex(Long)

    /**
     * Add a Long in hexadecimal notation to the log message.
     *
     * @param       val       The long to add.
     *
     * @return      The builder itself for chaining.
     */
    public T addHex(final Long val)
    {
        if (val != null) {
            return addHex(val.longValue());
        }
        sb.append("null");
        return (T) this;
    }

    // }}}
    // {{{ add(double)

    /**
     * Add a double to the log message.
     *
     * @param       val       The long to add.
     *
     * @return      The builder itself for chaining.
     */
    public T add(final double val)
    {
        sb.append(val);
        return (T) this;
    }

    // }}}
    // {{{ add(Object)

    /**
     * Add an Object to the log message. If the object is an array or iterable, the first 15 elements
     * are added to the log.
     *
     * @param       val       The Object to add.
     *
     * @return      The builder itself for chaining.
     */
    public T add(final Object val)
    {
        if (val == null) {
            sb.append(val);
        } else {
            try {
                if (val.getClass().isArray()) {
                    return addArray(val);
                } else if (val instanceof Iterable<?>) {
                    return addIterable((Iterable<?>) val);
                } else if (val instanceof Map<?,?>) {
                    return addMap((Map<?,?>) val);
                } else if (val instanceof Throwable) {
                    return addThrowable((Throwable) val);
                }

                if (! addUsingObjectDumper(val)) {
                    sb.append(val);
                }
            } catch (Throwable ex) {
                logger.log(Log.WRN,method, ex);
                sb.append("EXCEPTED");
            }
        }
        return (T) this;
    }

    // }}}
    // {{{ addUsingObjectDumper

    @NonNull
    private boolean addUsingObjectDumper(final Object val)
    {
        final LogObjectDumper dumper = getObjectDumperForValue(val);
        if (dumper != null && dumper.method.add(sb, val) != null) {
            return true;
        } else {
            return false;
        }
    }

    // }}}
    // {{{ getObjectDumperForValue

    @NonNull
    private LogObjectDumper getObjectDumperForValue(final Object val)
    {
        if (objectDumpers != null && val != null) {
            for (final LogObjectDumper dumper: objectDumpers) {
                if (dumper.clazz.isAssignableFrom(val.getClass())) {
                    return dumper;
                }
            }
        }

        return null;
    }

    // }}}
    // {{{ add(String)

    /**
     * Add a String to the log message.
     *
     * @param       val       The String to add.
     *
     * @return      The builder itself for chaining.
     */
    public T add(final String val)
    {
        if (objectDumpers != null && addUsingObjectDumper(val)) {
            return (T) this;
        }

        sb.append(val);

        return (T) this;
    }

    // }}}
    // {{{ add(Throwable)

    /**
     * Adds a Throwable to the log message. In addition, a second log call on detail level
     * is triggered containing the stack trace of the error.
     *
     * @param       throwable       The Throwable to add.
     *
     * @return      The builder itself for chaining.
     */
    protected T addThrowable(final Throwable throwable)
    {
        if (objectDumpers != null && addUsingObjectDumper(throwable)) {
            return (T) this;
        }

        sb.append(throwable.getClass().getName() +
                    ": " + throwable.getMessage());

        final String message = Log.getStackTrace(throwable);

        addTriggeredLogEntry(this.level + 1, formatMessage(message));

        return (T) this;
    }

    // }}}
    // {{{ addArray(Object)

    /**
     * Add the first 15 objects in an array of Objects to the log message.
     * Use {@link #setNumberOfArrayElementsToLog(int)} to change the
     * number of logged elements.
     *
     * @param       array       The objects to add.
     *
     * @return      The builder itself for chaining.
     */
    protected T addArray(final Object array)
    {
        assert array != null;
        assert array.getClass().isArray();

        if (objectDumpers != null && addUsingObjectDumper(array)) {
            return (T) this;
        }

        int arrayLength = Array.getLength(array);
        int hiddenElements = Math.max(0,arrayLength - numberOfArrayElementsToLog);

        sb.append("[");
        for (int i = 0; i < (arrayLength - hiddenElements); i++) {
            if (i > 0) {
                sb.append(',');
            }
            final Object element = Array.get(array, i);
            add(element == array ? "(this Array)" : element);
        }
        if (hiddenElements > 0) {
            sb.append(",... (")
              .append(hiddenElements)
              .append(" more)");
        }
        sb.append("]");
        return (T) this;
    }

    // }}}
    // {{{ addIterable(Iterable)

    /**
     * Add the first 30 objects in a Iterable of Objects to the log message.
     * Use {@link #setNumberOfArrayElementsToLog(int)} to change the
     * number of logged elements.
     *
     * @param       val       The objects to add.
     *
     * @return      The builder itself for chaining.
     */
    @SuppressWarnings("rawtypes")
    protected T addIterable(final Iterable val)
    {
        if (val == null) {
            sb.append(val);
        } else {

            if (objectDumpers != null && addUsingObjectDumper(val)) {
                return (T) this;
            }

            sb.append("[");
            int i = 0;
            try {
                for (Object element : val) {
                    if (i > 0) {
                        sb.append(',');
                    }
                    ++i;
                    if (i > numberOfArrayElementsToLog) {
                        sb.append("...");
                        if (val instanceof Collection) {
                            int listLength = ((Collection) val).size();
                            int hiddenElements = listLength - numberOfArrayElementsToLog;
                            sb.append(String.format(" (%d more)", hiddenElements));
                        }
                        break;
                    }
                    add(element == val ? "(this Iterable)" : element);
                }
            } catch (final ConcurrentModificationException ex) {
                sb.append(" ConcurrentModification!");
            }
            sb.append("]");
        }
        return (T) this;
    }
    // }}}
    // {{{ addMap(Map)

    /**
     * Add the first 50 objects in a Map of Objects to the log message.
     * Use {@link #setNumberOfArrayElementsToLog(int)} to change the
     * number of logged elements.
     *
     * @param       val       The objects to add.
     *
     * @return      The builder itself for chaining.
     */
    @SuppressWarnings("rawtypes")
    protected T addMap(final Map val)
    {
        if (val == null) {
            sb.append(val);
        } else {

            if (objectDumpers != null && addUsingObjectDumper(val)) {
                return (T) this;
            }

            sb.append("{");
            try {
                int i = 0;
                final Set<Entry<Object,Object>> entrySet = val.entrySet();
                for(final Entry<Object,Object> entry: entrySet) {
                    if (i > 0) {
                        sb.append(',');
                    }
                    ++i;
                    if (i > numberOfMapElementsToLog) {
                        sb.append("...");
                        int listLength = entrySet.size();
                        int hiddenElements = listLength - numberOfMapElementsToLog;
                        sb.append(String.format(" (%d more)",hiddenElements));
                        break;
                    }

                    final Object key = entry.getKey();
                    final Object value = entry.getValue();

                    add(key == val ? "(this Map)" : key);
                    sb.append("=");
                    add(value == val ? "(this Map)" : value);
                }
            } catch (final ConcurrentModificationException ex) {
                sb.append(" ConcurrentModification!");
            }
            sb.append("}");
        }
        return (T) this;
    }
    // }}}
    // {{{ addAll(Object)

    /**
     * If the object is an array, add all of its elements, otherwise add the object itself.
     *
     * @param       val       The object(s) to add.
     *
     * @return      The builder itself for chaining.
     */
    @SuppressWarnings("rawtypes")
    public T addAll(final Object val)
    {
        synchronized(addAllMonitor) {
            int previousNumberOfArrayElementsToLog = this.numberOfArrayElementsToLog;
            int previousNumberOfMapElementsToLog = this.numberOfMapElementsToLog;
            return setNumberOfArrayElementsToLog(Integer.MAX_VALUE)
                    .setNumberOfMapElementsToLog(Integer.MAX_VALUE)
                    .add(val)
                    .setNumberOfMapElementsToLog(previousNumberOfMapElementsToLog)
                    .setNumberOfArrayElementsToLog(previousNumberOfArrayElementsToLog);
        }
    }

    // }}}
    // {{{ addLabel(String)

    /**
     * Adds a label to the Logmessage.
     *
     * @param       label     The value's label
     *
     */
    protected void addLabel(final String label)
    {
        if (sb.length() > 0) {
            sb.append(", ");
        }
        if (label == null) {
            sb.append(label);
        } else {
            sb.append(label.replaceAll("^\\s*,?\\s*|\\s*:?\\s*$", ""));
        }
        sb.append(": ");
    }

    // }}}
    // {{{ add(String, boolean)

    /**
     * Add a boolean to the log message.
     *
     * @param       label     The value's label
     * @param       val       The boolean to add.
     *
     * @return      The builder itself for chaining.
     */
    public T add(final String label, final boolean val)
    {
        addLabel(label);
        return add(val);
    }

    // }}}
    // {{{ add(String, byte)

    /**
     * Add a byte to the log message.
     *
     * @param       label     The value's label
     * @param       val       The byte to add.
     *
     * @return      The builder itself for chaining.
     */
    public T add(final String label, final byte val)
    {
        addLabel(label);
        return add(val);
    }

    // }}}
    // {{{ add(String, char)

    /**
     * Add a char to the log message.
     *
     * @param       label     The value's label
     * @param       val       The char to add.
     *
     * @return      The builder itself for chaining.
     */
    public T add(final String label, final char val)
    {
        addLabel(label);
        return add(val);
    }

    // }}}
    // {{{ add(String, int)

    /**
     * Add an int to the log message.
     *
     * @param       label     The value's label
     * @param       val       The int to add.
     *
     * @return      The builder itself for chaining.
     */
    public T add(final String label, final int val)
    {
        addLabel(label);
        return add(val);
    }

    // }}}
    // {{{ addHex(String, int)

    /**
     * Add an int in hexadecimal notation to the log message.
     *
     * @param       label     The value's label
     * @param       val       The int to add.
     *
     * @return      The builder itself for chaining.
     */
    public T addHex(final String label, final int val)
    {
        addLabel(label);
        return addHex(val);
    }

    // }}}
    // {{{ addHex(String, Integer)

    /**
     * Add an Integer in hexadecimal notation to the log message.
     *
     * @param       label     The value's label
     * @param       val       The int to add.
     *
     * @return      The builder itself for chaining.
     */
    public T addHex(final String label, final Integer val)
    {
        addLabel(label);
        return addHex(val);
    }

    // }}}
    // {{{ add(String, long)

    /**
     * Add a long to the log message.
     *
     * @param       label     The value's label
     * @param       val       The long to add.
     *
     * @return      The builder itself for chaining.
     */
    public T add(final String label, final long val)
    {
        addLabel(label);
        return add(val);
    }

    // }}}
    // {{{ addHex(String, long)

    /**
     * Add an long in hexadecimal notation to the log message.
     *
     * @param       label     The value's label
     * @param       val       The long to add.
     *
     * @return      The builder itself for chaining.
     */
    public T addHex(final String label, final long val)
    {
        addLabel(label);
        return addHex(val);
    }

    // }}}
    // {{{ addHex(String, Long)

    /**
     * Add a Long in hexadecimal notation to the log message.
     *
     * @param       label     The value's label
     * @param       val       The long to add.
     *
     * @return      The builder itself for chaining.
     */
    public T addHex(final String label, final Long val)
    {
        addLabel(label);
        return addHex(val);
    }

    // }}}
    // {{{ add(String, double)

    /**
     * Add a double to the log message.
     *
     * @param       label     The value's label
     * @param       val       The long to add.
     *
     * @return      The builder itself for chaining.
     */
    public T add(final String label, final double val)
    {
        addLabel(label);
        return add(val);
    }

    // }}}
    // {{{ add(String, Object)

    /**
     * Add an Object to the log message. If the object is an array, the first 15 elements
     * are added to the log. Use {@link #setNumberOfArrayElementsToLog(int)} to change the
     * number of logged elements.
     *
     * @param       label     The value's label
     * @param       val       The Object to add.
     *
     * @return      The builder itself for chaining.
     */
    public T add(final String label, final Object val)
    {
        addLabel(label);
        return add(val);
    }

    // }}}
    // {{{ add(String, String)

    /**
     * Add a String to the log message.
     *
     * @param       label     The value's label
     * @param       val       The String to add.
     *
     * @return      The builder itself for chaining.
     */
    public T add(final String label, final String val)
    {
        addLabel(label);
        return add(val);
    }

    // }}}
    // {{{ addAll(String, Object)

    /**
     * If the object is an array, add all of its elements, otherwise add the object itself.
     *
     * @param       label     The value's label
     * @param       val       The object(s) to add.
     *
     * @return      The builder itself for chaining.
     */
    public T addAll(final String label, final Object val)
    {
        addLabel(label);
        return addAll(val);
    }

    // }}}

    // {{{ addDetail(boolean)

    /**
     * Add a boolean to the log message, if the enclosing {@link Logger}'s
     * current log level is at least the Builder's level + 1.
     *
     * @param val
     *            The boolean to add.
     *
     * @return The builder itself for chaining.
     */
    public T addDetail(final boolean val)
    {
        if (useDetails) {
            add(val);
        }
        return (T) this;
    }

    // }}}
    // {{{ addDetail(byte)

    /**
     * Add a byte to the log message, if the enclosing {@link Logger}'s current
     * log level is at least the Builder's level + 1.
     *
     * @param val
     *            The byte to add.
     *
     * @return The builder itself for chaining.
     */
    public T addDetail(final byte val)
    {
        if (useDetails) {
            add(val);
        }
        return (T) this;
    }

    // }}}
    // {{{ addDetail(char)

    /**
     * Add a char to the log message, if the enclosing {@link Logger}'s current
     * log level is at least the Builder's level + 1.
     *
     * @param val
     *            The char to add.
     *
     * @return The builder itself for chaining.
     */
    public T addDetail(final char val) {
        if (useDetails) {
            add(val);
        }
        return (T) this;
    }

    // }}}
    // {{{ addDetail(int)

    /**
     * Add an int to the log message, if the enclosing {@link Logger}'s current
     * log level is at least the Builder's level + 1.
     *
     * @param val
     *            The int to add.
     *
     * @return The builder itself for chaining.
     */
    public T addDetail(final int val) {
        if (useDetails) {
            add(val);
        }
        return (T) this;
    }

    // }}}
    // {{{ addHexDetail(int)

    /**
     * Add an int in hexadecimal notation to the log message, if the enclosing
     * {@link Logger}'s current log level is at least the Builder's level + 1.
     *
     * @param val
     *            The int to add.
     *
     * @return The builder itself for chaining.
     */
    public T addHexDetail(final int val) {
        if (useDetails) {
            addHex(val);
        }
        return (T) this;
    }

    // }}}
    // {{{ addHexDetail(Integer)

    /**
     * Add an Integer in hexadecimal notation to the log message, if the
     * enclosing {@link Logger}'s current log level is at least the Builder's
     * level + 1.
     *
     * @param val
     *            The int to add.
     *
     * @return The builder itself for chaining.
     */
    public T addHexDetail(final Integer val) {
        if (useDetails) {
            addHex(val);
        }
        return (T) this;
    }

    // }}}
    // {{{ addDetail(long)

    /**
     * Add a long to the log message, if the enclosing {@link Logger}'s current
     * log level is at least the Builder's level + 1.
     *
     * @param val
     *            The long to add.
     *
     * @return The builder itself for chaining.
     */
    public T addDetail(final long val) {
        if (useDetails) {
            add(val);
        }
        return (T) this;
    }

    // }}}
    // {{{ addHexDetail(long)

    /**
     * Add an long in hexadecimal notation to the log message, if the enclosing
     * {@link Logger}'s current log level is at least the Builder's level + 1.
     *
     * @param val
     *            The long to add.
     *
     * @return The builder itself for chaining.
     */
    public T addHexDetail(final long val) {
        if (useDetails) {
            addHex(val);
        }
        return (T) this;
    }

    // }}}
    // {{{ addHexDetail(Long)

    /**
     * Add a Long in hexadecimal notation to the log message, if the enclosing
     * {@link Logger}'s current log level is at least the Builder's level + 1.
     *
     * @param val
     *            The long to add.
     *
     * @return The builder itself for chaining.
     */
    public T addHexDetail(final Long val) {
        if (useDetails) {
            addHex(val);
        }
        return (T) this;
    }

    // }}}
    // {{{ addDetail(double)

    /**
     * Add a double to the log message, if the enclosing {@link Logger}'s
     * current log level is at least the Builder's level + 1.
     *
     * @param val
     *            The long to add.
     *
     * @return The builder itself for chaining.
     */
    public T addDetail(final double val) {
        if (useDetails) {
            add(val);
        }
        return (T) this;
    }

    // }}}
    // {{{ addDetail(Object)

    /**
     * Add an Object to the log message, if the enclosing {@link Logger}'s
     * current log level is at least the Builder's level + 1.
     *
     * @param val
     *            The Object to add.
     *
     * @return The builder itself for chaining.
     */
    public T addDetail(final Object val) {
        if (useDetails) {
            add(val);
        }
        return (T) this;
    }

    // }}}
    // {{{ addDetail(String)

    /**
     * Add a String to the log message, if the enclosing {@link Logger}'s
     * current log level is at least the Builder's level + 1.
     *
     * @param val
     *            The String to add.
     *
     * @return The builder itself for chaining.
     */
    public T addDetail(final String val) {
        if (useDetails) {
            add(val);
        }
        return (T) this;
    }

    // }}}
    // {{{ addDetailLabel(String)

    /**
     * Adds a label to the Log message.
     *
     * @param label
     *            The value's label
     *
     */
    protected void addDetailLabel(final String label) {
        if (useDetails) {
            addLabel(label);
        }
    }

    // }}}
    // {{{ addDetail(String, boolean)

    /**
     * Add a boolean to the log message, if the enclosing {@link Logger}'s
     * current log level is at least the Builder's level + 1.
     *
     * @param label
     *            The value's label
     * @param val
     *            The boolean to add.
     *
     * @return The builder itself for chaining.
     */
    public T addDetail(final String label, final boolean val) {
        addDetailLabel(label);
        return addDetail(val);
    }

    // }}}
    // {{{ addDetail(String, byte)

    /**
     * Add a byte to the log message, if the enclosing {@link Logger}'s current
     * log level is at least the Builder's level + 1.
     *
     * @param label
     *            The value's label
     * @param val
     *            The byte to add.
     *
     * @return The builder itself for chaining.
     */
    public T addDetail(final String label, final byte val) {
        addDetailLabel(label);
        return addDetail(val);
    }

    // }}}
    // {{{ addDetail(String, char)

    /**
     * Add a char to the log message, if the enclosing {@link Logger}'s current
     * log level is at least the Builder's level + 1.
     *
     * @param label
     *            The value's label
     * @param val
     *            The char to add.
     *
     * @return The builder itself for chaining.
     */
    public T addDetail(final String label, final char val) {
        addDetailLabel(label);
        return addDetail(val);
    }

    // }}}
    // {{{ addDetail(String, int)

    /**
     * Add an int to the log message, if the enclosing {@link Logger}'s current
     * log level is at least the Builder's level + 1.
     *
     * @param label
     *            The value's label
     * @param val
     *            The int to add.
     *
     * @return The builder itself for chaining.
     */
    public T addDetail(final String label, final int val) {
        addDetailLabel(label);
        return addDetail(val);
    }

    // }}}
    // {{{ addHexDetail(String, int)

    /**
     * Add an int in hexadecimal notation to the log message, if the enclosing
     * {@link Logger}'s current log level is at least the Builder's level + 1.
     *
     * @param label
     *            The value's label
     * @param val
     *            The int to add.
     *
     * @return The builder itself for chaining.
     */
    public T addHexDetail(final String label, final int val) {
        addDetailLabel(label);
        return addHexDetail(val);
    }

    // }}}
    // {{{ addHexDetail(String, Integer)

    /**
     * Add an Integer in hexadecimal notation to the log message, if the
     * enclosing {@link Logger}'s current log level is at least the Builder's
     * level + 1.
     *
     * @param label
     *            The value's label
     * @param val
     *            The int to add.
     *
     * @return The builder itself for chaining.
     */
    public T addHexDetail(final String label, final Integer val) {
        addDetailLabel(label);
        return addHexDetail(val);
    }

    // }}}
    // {{{ addDetail(String, long)

    /**
     * Add a long to the log message, if the enclosing {@link Logger}'s current
     * log level is at least the Builder's level + 1.
     *
     * @param label
     *            The value's label
     * @param val
     *            The long to add.
     *
     * @return The builder itself for chaining.
     */
    public T addDetail(final String label, final long val) {
        addDetailLabel(label);
        return addDetail(val);
    }

    // }}}
    // {{{ addHexDetail(String, long)

    /**
     * Add an long in hexadecimal notation to the log message, if the enclosing
     * {@link Logger}'s current log level is at least the Builder's level + 1.
     *
     * @param label
     *            The value's label
     * @param val
     *            The long to add.
     *
     * @return The builder itself for chaining.
     */
    public T addHexDetail(final String label, final long val) {
        addDetailLabel(label);
        return addHexDetail(val);
    }

    // }}}
    // {{{ addHexDetail(String, Long)

    /**
     * Add a Long in hexadecimal notation to the log message, if the enclosing
     * {@link Logger}'s current log level is at least the Builder's level + 1.
     *
     * @param label
     *            The value's label
     * @param val
     *            The long to add.
     *
     * @return The builder itself for chaining.
     */
    public T addHexDetail(final String label, final Long val) {
        addDetailLabel(label);
        return addHexDetail(val);
    }

    // }}}
    // {{{ addDetail(String, double)

    /**
     * Add a double to the log message, if the enclosing {@link Logger}'s
     * current log level is at least the Builder's level + 1.
     *
     * @param label
     *            The value's label
     * @param val
     *            The long to add.
     *
     * @return The builder itself for chaining.
     */
    public T addDetail(final String label, final double val) {
        addDetailLabel(label);
        return addDetail(val);
    }

    // }}}
    // {{{ addDetail(String, Object)

    /**
     * Add an Object to the log message, if the enclosing {@link Logger}'s
     * current log level is at least the Builder's level + 1.
     *
     * @param label
     *            The value's label
     * @param val
     *            The Object to add.
     *
     * @return The builder itself for chaining.
     */
    public T addDetail(final String label, final Object val) {
        addDetailLabel(label);
        return addDetail(val);
    }

    // }}}
    // {{{ addDetail(String, String)

    /**
     * Add a String to the log message, if the enclosing {@link Logger}'s
     * current log level is at least the Builder's level + 1.
     *
     * @param label
     *            The value's label
     * @param val
     *            The String to add.
     *
     * @return The builder itself for chaining.
     */
    public T addDetail(final String label, final String val) {
        addDetailLabel(label);
        return addDetail(val);
    }

    // }}}

    // {{{ log()

    /**
     * Log the message, using the Builder's log level.
     */
    public void log() {
        logAt(System.currentTimeMillis());
    }


    // }}}

    // {{{ logAt()

    /**
     * Log the message, using the Builder's log level.
     */
    public void logAt(final long timestamp) {
        logger.log(this.level, timestamp,
                this.method, formatMessage(getMessage()));
        dispatchTriggeredLogEntries();
    }


    // }}}
    // {{{ formatMessage(String)

    protected String formatMessage(final String message) {
        return lineNumber > 0 ? "(#" + lineNumber + ")" +
                                    (message.length() > 0 ? " " +message
                                                     : "")
                              : message;
    }

    // }}}
    // {{{ getMessage()

    protected String getMessage() {
        return sb.toString();
    }

    // }}}
    // {{{ dumpStack()

    /**
     * Log the message including a stack dump, using the Builder's log level.
     */
    public void dumpStack() {
        logger.dumpStack(this.level, method, formatMessage(getMessage()));
        dispatchTriggeredLogEntries();
    }

    // }}}
    // {{{ addTriggeredLogEntry(int,String)

    protected void addTriggeredLogEntry(final int level, final String message) {
        if (triggeredLogEntries == null) {
            triggeredLogEntries = new ConcurrentLinkedQueue<LogEntry>();
        }
        triggeredLogEntries.add(new LogEntry(level, System.currentTimeMillis(), null, null, method, message));
    }

    // }}}
    // {{{ dispatchTriggeredLogEntries()

    protected void dispatchTriggeredLogEntries() {
        if (triggeredLogEntries == null) {
            return;
        }
        for (final LogEntry entry: triggeredLogEntries) {
            logger.log(entry.getLevel(),entry.getTimestamp(),entry.getMethod(),entry.getMessage());
        }
    }

    // }}}

}

// }}}