// {{{ copyright

/********************************************************************
 *
 * The contents of this file are subject to the Mozilla Public
 * License Version 1.1 (the "License"); you may not use this file
 * except in compliance with the License. You may obtain a copy of
 * the License at http://www.mozilla.org/MPL/
 *
 * Software distributed under the License is distributed on an "AS
 * IS" basis, WITHOUT WARRANTY OF ANY KIND, either express or
 * implied. See the License for the specific language governing
 * rights and limitations under the License.
 *
 * The Original Code is qfs.de code.
 *
 * The Initial Developer of the Original Code is Gregor Schmid.
 * Portions created by Gregor Schmid are
 * Copyright (C) 2000 Quality First Software, Gregor Schmid.
 * All Rights Reserved.
 *
 * Contributor(s):
 *
 *******************************************************************/

// }}}

package de.qfs.lib.log;

// {{{ doc

/**
 * A simple {@link LogFilter LogFilter} that filters messages based on their
 * level. <p>
 *
 * It considers all messages with a level less than or equal to the given
 * maximum as useful. If a {@link LogWriter LogWriter} has been set, all
 * useful messages will be written to it. By setting the maximum level to a
 * value < 0, filtering can be turned off, so all messages are passed to the
 * writer. <p>
 *
 * Whether a message is passed on through the filter chain depends on whether
 * it is considered useful and on the mode setting as described for the {@link
 * LogUser LogUser} interface.
 *
 * @author      Gregor Schmid
 * @since       0.98.0
 */

// }}}
public class LevelFilter
    extends AbstractLogUser
{
    // {{{ variables

    /**
     * The maximum level to output.
     */
    private int level = -1;

    // }}}

    //----------------------------------------------------------------------
    // Constructors
    //----------------------------------------------------------------------
    // {{{ LevelFilter(int)

    /**
     * Create a new LevelFilter with mode {@link LogUser#FILTER_UNUSED
     * FILTER_UNUSED} and no {@link LogWriter LogWriter}.
     *
     * @param   level   Only messages up to the returned level will be passed
     *                  on and possibly written to the LogWriter.
     */
    public LevelFilter(int level)
    {
        super(FILTER_UNUSED, null);
        this.level = level;
    }

    // }}}
    // {{{ LevelFilter(int,LogWriter)

    /**
     * Create a new LevelFilter with mode {@link LogUser#FILTER_USED
     * FILTER_USED}.
     *
     * @param   level   Only messages up to the returned level will be passed
     *                  on and possibly written to the LogWriter.
     * @param   writer  The LogWriter used to write LogEntries that pass the
     *                  filter setting.
     */
    public LevelFilter(int level, LogWriter writer)
    {
        super(FILTER_NONE, writer);
        this.level = level;
    }

    // }}}
    // {{{ LevelFilter(int,int,LogWriter)

    /**
     * Create a new LevelFilter.
     *
     * @param   level   Only messages up to the returned level will be passed
     *                  on and possibly written to the LogWriter.
     * @param   mode    The {@link LogUser LogUser} mode.
     * @param   writer  The LogWriter used to write LogEntries that pass the
     *                  filter setting.
     */
    public LevelFilter(int level, int mode, LogWriter writer)
    {
        super(mode, writer);
        this.level = level;
    }

    // }}}

    //----------------------------------------------------------------------
    // Public methods.
    //----------------------------------------------------------------------
    // {{{ getLevel

    /**
     * Get the level limiting the filtering and writing of messages.
     *
     * @return  Only messages up to the returned level will be passed on and
     *          possibly written to the LogWriter.
     */
    public final int getLevel()
    {
        return level;
    }

    // }}}
    // {{{ setLevel

    /**
     * Set the level limiting the filtering and writing of messages.
     *
     * @param   level   Only messages up to the returned level will be passed
     *                  on and possibly written to the LogWriter.
     */
    public final void setLevel(int level)
    {
        this.level = level;
    }

    // }}}

    //----------------------------------------------------------------------
    // The LogUser interface.
    //----------------------------------------------------------------------
    // {{{ filter(LogEntry[])

    /**
     * Filter an array of LogEntries, possibly writing some of them to the
     * LogWriter. This method is overridden to try some time and memory-saving
     * shortcuts.
     *
     * @param   entries The entries to filter.
     */
    @Override
    public LogEntry[] filter (LogEntry[] entries)
    {
        // First shortcut
        if (level < 0) {
            if (getLogWriter() != null) {
                getLogWriter().write(entries);
            }
            if (mode == FILTER_ALL || mode == FILTER_USED) {
                return null;
            }
            return entries;
        }

        // If there's no writer, there may be another shortcut
        if (getLogWriter() == null) {
            if (mode == FILTER_ALL) {
                return null;
            }
            if (mode == FILTER_NONE) {
                return entries;
            }
        }

        // Its up to the base class.
        return super.filter(entries);
    }

    // }}}

    //----------------------------------------------------------------------
    // The actual filter method
    //----------------------------------------------------------------------
    // {{{ isUseful

    /**
     * Decide whether a log message is useful and should be passed to the log
     * writer.
     *
     * @param   entry   The LogEntry to check.
     *
     * @return True if the entry's level is less than or equal to the
     * LevelFilter's level. It the LevelFilter's level is -1, filtering is
     * turned off and the result is always true.
     */
    @Override
    protected boolean isUseful(LogEntry entry)
    {
        return level == -1 || entry.getLevel() <= level;
    }

    // }}}
}
