// {{{ copyright

/********************************************************************
 *
 * The contents of this file are subject to the Mozilla Public
 * License Version 1.1 (the "License"); you may not use this file
 * except in compliance with the License. You may obtain a copy of
 * the License at http://www.mozilla.org/MPL/
 *
 * Software distributed under the License is distributed on an "AS
 * IS" basis, WITHOUT WARRANTY OF ANY KIND, either express or
 * implied. See the License for the specific language governing
 * rights and limitations under the License.
 *
 * The Original Code is qfs.de code.
 *
 * The Initial Developer of the Original Code is Gregor Schmid.
 * Portions created by Gregor Schmid are
 * Copyright (C) 1999 Quality First Software, Gregor Schmid.
 * Portions created by Oliver Brandt are
 * Copyright (C) 2000 ATecoM GmbH.
 * All Rights Reserved.
 *
 * Contributor(s):
 * Oliver Brandt, ATecoM GmbH
 *
 *******************************************************************/

// }}}

package de.qfs.lib.log;

//{{{ imports

import java.io.PrintWriter;
import java.io.StringWriter;

import java.util.Hashtable;
import java.util.Vector;

// }}}

// {{{ doc

/**
 * The Log class coordiantes the logging of error and debug messages of
 * different levels of importance. Currently 10 different levels of importance
 * are defined, ranging from ERR for errors to DBGDETAIL for detailed
 * debugging messages. <p>
 *
 * Apart from the messages themselves, the sending class and method will be
 * noted, as well as the current time and the active Thread.<p>
 *
 * Log messages are best dispatched with the help of the {@link Logger Logger}
 * class. <p>
 *
 * Once logging is used seriously, the number of logged messages tends to grow
 * rapidly. There are various ways to filter out messages, while still keeping
 * important information, to reduce the impact of logging on CPU time and
 * memory consumption. <p>
 *
 * The most expensive part of creating log messages is the construction of the
 * message String, which often involves calls of toString() methods or
 * concatenation. This cost can only be reduced by protecting the log call
 * with some flag or level checking mechanism. A very flexible scheme that
 * allows simple configuration of these levels as well as optional runtime
 * modification is provided by the {@link Logger Logger} class. <p>
 *
 * Once messages are created, the cheapest and most effective method to reduce
 * logging is to set the pre-queue filter level with {@link #setPreQueueLevel
 * setPreQueueLevel}. Messages with a level higher than the pre-queue level
 * will be (almost) instantly discarded. <p>
 *
 * Messages passing the pre-queue filter mechanism can either be queued to be
 * processed by a low level background thread at a later time, or passed on
 * immediately. This is controlled with the {@link #setQueueing setQueueing}
 * method. Queueing will reduce logging overhead drastically, but can increase
 * memory consumption if the program is running constantly on a high load. <p>
 *
 * Queueing should be turned off before terminating a program, so that all
 * pending messages will be flushed.<p>
 *
 * Once messages really get dispatched, they will be passed through a chain of
 * filters, that must implement the {@link LogFilter LogFilter} or the {@link
 * LogUser LogUser} interface. These filters are intended primarily for
 * diverting the messages to a log file or a remote log server. <p>
 *
 * Finally, if a message is still not filtered out, its level is checked
 * against the current print level set with {@link #setOutputLevel
 * setOutputLevel} and, if its level is less than or equal to the print level,
 * it is passed to a {@link LogWriter LogWriter} that can be cutomized with
 * {@link #setLogWriter setLogWriter}. The default writer used prints the
 * messages to System.err. <p>
 *
 * To complicate things further, there is an additional mechanism. The
 * pre-queue filter level is most useful when set rather low, so only error
 * messages will pass. But if an error happens, the information needed to find
 * out why is not available. To get at that information, a flush buffer can be
 * used. <p>
 *
 * The flush buffer will save the last n messages that were filtered out
 * during the pre-queue stage, where n can be set with {@link
 * #setFlushBufferSize setFlushBufferSize}. When a message is logged that has
 * a level less than or equal to a trigger level set with {@link
 * #setFlushTriggerLevel setFlushTriggerLevel}, all saved messages will be
 * recovered from the flush buffer und flushed onto the queue. <p>
 *
 * Since valuable information may also be logged after an error happened, the
 * number of messages set with {@link #setPostFlushSize setPostFlushSize} will
 * pass unfiltered through the pre-queue stage after a flush was
 * triggered. <p>
 *
 * Yet another filter mechanism is used for completely different reasons. If a
 * LogFilter causes more messages to be logged, endless recursion or deadlocks
 * are almost inevitable. To avoid those, filters that are creating log
 * messages themselves, or call methods that may use logging, must make sure
 * they call {@link #excludeThread excludeThread} before they start their work
 * and {@link #includeThread includeThread} when they are finished, preferably
 * in a finally clause. <p>
 *
 * @author      Gregor Schmid
 */

// }}}
public class Log
{
    // {{{ Error level constants

    /**
     * The level for error messages and other messages that should
     * always be logged.
     */
    public static final int ERR = 1;

    /**
     * The level for details for error messages.
     */
    public static final int ERRDETAIL = 2;

    /**
     * The level for warnings.
     */
    public static final int WRN = 3;

    /**
     * The level for details for warnings.
     */
    public static final int WRNDETAIL = 4;

    /**
     * The level for normal messages of medium importance.
     */
    public static final int MSG = 5;

    /**
     * The level for details for normal messages.
     */
    public static final int MSGDETAIL = 6;

    /**
     * The level for method calls.
     */
    public static final int MTD = 7;

    /**
     * The level for details for Method calls i.e. detailed parameter values.
     */
    public static final int MTDDETAIL = 8;

    /**
     * The level for debugging messages.
     */
    public static final int DBG = 9;

    /**
     * The level for details for debugging messages. These can sometimes
     * produce an enormous amount of data.
     */
    public static final int DBGDETAIL = 10;

    // }}}

    // {{{ variables

    private static boolean DEBUG = "true".equalsIgnoreCase(System.getenv("QFLIB_LOG_DEBUG"));

    /**
     * The default value for the output level: DBGDETAIL.
     */
    public final static int DEFAULT_OUTPUT_LEVEL = DBGDETAIL;

    /**
     * Lock for static synchronization.
     */
    private final static Object classlock = new Object ();

    /**
     * The output log level.
     */
    private static int level = DEFAULT_OUTPUT_LEVEL;

    /**
     * The default value for the pre queue level: DBGDETAIL.
     */
    public final static int DEFAULT_PRE_QUEUE_LEVEL = DBGDETAIL;

    /**
     * The pre-queue log level.
     */
    private static int preQueueLevel = DEFAULT_PRE_QUEUE_LEVEL;

    /**
     * The default queueing flag: true.
     */
    public final static boolean DEFAULT_QUEUEING = true;

    /**
     * The default initial queue size: 30000.
     */
    public final static int DEFAULT_INITIAL_QUEUE_SIZE = 30000;

    /**
     * The default queue size: 3000.
     */
    public final static int DEFAULT_QUEUE_SIZE = 3000;

    /**
     * The default initial drop on overflow flag: true.
     */
    public final static boolean DEFAULT_INITIAL_DROP_ON_OVERFLOW = true;

    /**
     * The default drop on overflow flag: false.
     */
    public final static boolean DEFAULT_DROP_ON_OVERFLOW = false;

    /**
     * The default size of the flush buffer: 0.
     */
    public final static int DEFAULT_FLUSH_BUFFER_SIZE = 0;

    /**
     * The flush buffer.
     */
    private static FlushBuffer flushBuffer =
        DEFAULT_FLUSH_BUFFER_SIZE == 0 ? null
        : new FlushBuffer (DEFAULT_FLUSH_BUFFER_SIZE);

    /**
     * The default value for the flush trigger level: ERR.
     */
    public final static int DEFAULT_FLUSH_TRIGGER_LEVEL = ERR;

    /**
     * The flush trigger level
     */
    private static int flushTriggerLevel = DEFAULT_FLUSH_TRIGGER_LEVEL;

    /**
     * The default value for the post flush size: 10.
     */
    public final static int DEFAULT_POST_FLUSH_SIZE = 10;

    /**
     * Number of messages to pass after a flush was triggered
     */
    private static int postFlushSize = DEFAULT_POST_FLUSH_SIZE;

    /**
     * Counter for logged messages after a flush was triggered
     */
    private static int postFlushCount = 0;

    /**
     * Default maximum size for messages to log.
     */
    public final static int DEFAULT_MAXIMUM_MESSAGE_SIZE = 20000;

    /**
     * Maximum size for messages to log. Longer messages will be truncated. Set <= 0 to disable
     * truncation.
     */
    private static int maxMessageSize = DEFAULT_MAXIMUM_MESSAGE_SIZE;

    /**
     * Message queue
     */
    private static LogQueue queue = new LogQueue();

    /**
     * Whether logs are being queued.
     */
    private static boolean queueing = true;

    /**
     * Queue size to set after the LogThread has started with the initial flush.
     */
    private static int targetQueueSize = DEFAULT_QUEUE_SIZE;

    /**
     * Queue dropOnOverflow state to set after the LogThread has started with the initial flush.
     */
    private static boolean targetQueueDropOnOverflow = DEFAULT_DROP_ON_OVERFLOW;

    /**
     * The filter chain.
     */
    private static Vector filters = new Vector ();

    /**
     * The filter chain as an array.
     */
    private static LogFilter[] _filters = new LogFilter[0];

    /**
     * The writer for the default mechanism.
     */
    private static LogWriter writer = new StreamLogWriter ();

    /**
     * Threads that are excluded from logging at the moment.
     */
    private static Hashtable silenced = new Hashtable ();

    /**
     * The LogThread running in the background.
     */
    private static LogThread logThread = new LogThread();

    /**
     * Whether the LogThread has been started.
     */
    private static boolean logThreadStarted;

    static {
        silenced.put(logThread, Integer.valueOf(1));
        long delayLogThread = 500;
        try {
            delayLogThread = Long.parseLong(System.getenv("LOG_DELAY_LOG_THREAD"));
        } catch (Throwable ex) {
        }
        if (delayLogThread > 0) {
            final long delay = delayLogThread;
            Thread bgThread = new Thread("delay log thread") {
                public void run() {
                    try {
                        if (DEBUG) {
                            System.err.println("Log - now: " + System.currentTimeMillis()); // checkcode:ignore
                            System.err.println("Log - delay: " + delay); // checkcode:ignore
                        }
                        Thread.sleep(delay);
                    } catch (InterruptedException ex) {
                        // ignore
                    }
                    startLogThread();
                }
            };
            bgThread.setDaemon(true);
            bgThread.start();
        } else {
            startLogThread();
        }
    }

    // }}}

    //----------------------------------------------------------------------
    // Start the LogThread
    //----------------------------------------------------------------------
    // {{{ startLogThread

    /**
     * Start the LogThread - call when the primary filters have been
     * registered. Pops all entries from the queue and resets its level from
     * the initial size to the default.
     */
    public static void startLogThread()
    {
        synchronized (logThread) {
            if (! logThreadStarted) {
                if (DEBUG) {
                    System.err.println("startLogThread - now: " + System.currentTimeMillis()); // checkcode:ignore
                    Thread.dumpStack(); // checkcode:ignore
                }
                logThreadStarted = true;
                logThread.start();
            }
        }
    }


    // }}}

    //----------------------------------------------------------------------
    // Log messages and exceptions
    //----------------------------------------------------------------------
    // {{{ log(int,long,String,String,String)

    /**
     * Log a message.
     *
     * @param   level   The level of the message.
     * @param   timestamp The time when the message originated.
     * @param   clazz   The name of the sending class.
     * @param   method  The name of the sending method.
     * @param   message The message.
     */
    public static void log(int level, long timestamp,
                           String clazz, String method, String message)
    {
        Thread thread = Thread.currentThread();

        // Avoid recursive generation of message through the LogThread
        if (silenced.containsKey(thread)) {
            return;
        }
        log(level, timestamp, clazz, method, message, thread.getName());
    }

    // }}}
    // {{{ log(int,long,String,String,String,String)

    /**
     * Log a message.
     *
     * @param   level   The level of the message.
     * @param   timestamp The time when the message originated.
     * @param   clazz   The name of the sending class.
     * @param   method  The name of the sending method.
     * @param   message The message.
     * @param   treadName   The name of the originating thread.
     */
    public static void log(int level, long timestamp,
                           String clazz, String method, String message, String threadName)
    {
        Thread thread = Thread.currentThread();

        // Avoid recursive generation of message through the LogThread
        if (silenced.containsKey(thread)) {
            return;
        }

        if (maxMessageSize > 0
            && message != null
            && message.length() > maxMessageSize) {
            message = message.substring(0, maxMessageSize);
        }

        boolean postFlush = (postFlushCount > 0);
        if (postFlush) {
            // skip pre-queue checking
            postFlushCount--;
        } else {
            /**
             * first check is against the pre-queue level
             */
            if (level > preQueueLevel) {
                if (flushBuffer != null) {
                    flushBuffer.buffer(level, timestamp, clazz, method, message);
                }
                return;
            }
        }

        if (level <= flushTriggerLevel) {
            postFlushCount = postFlushSize;
            if (!postFlush && flushBuffer != null) {
                LogEntry[] entries = flushBuffer.flush();
                for (int i = 0; i < entries.length; i++) {
                    if (queueing) {
                        queue.push(entries[i]);
                    } else {
                        filter(entries[i]);
                    }
                }
            }
        }

        if (queueing) {
            queue.push(new LogEntry (level, timestamp, threadName, clazz, method, message));
        } else {
            filter(new LogEntry (level, timestamp, threadName, clazz, method, message));
        }
    }

    // }}}
    // {{{ log(int,String,String,Throwable)

    /**
     * Log an exception.
     *
     * @param   level   The level of the message.
     * @param   clazz   The name of the sending class.
     * @param   method  The name of the sending method.
     * @param   throwable The Throwable.
     */
    public static void log(int level, String clazz,
                           String method, Throwable throwable)
    {
        long now = System.currentTimeMillis();

        final boolean onlyDumpStacktrace = (throwable instanceof DumpStackException);

        // log the stacktrace
        String stackTrace = getStackTrace(throwable);

        String message = throwable.getClass().getName() + ": " + throwable.getMessage();

        if (onlyDumpStacktrace) {

            message = throwable.getMessage();

            // Remove leading exception Name from stacktrace
            int colonIndex = stackTrace.indexOf(':');
            if (colonIndex > -1 && stackTrace.length() >= colonIndex + 2) {
                stackTrace = stackTrace.substring(colonIndex + 2);
            }

        }

        // log the name and the message
        log(level, now, clazz, method, message);
        log(level + 1, now, clazz, method, stackTrace);
    }

    // }}}
    // {{{ log(int,String,String,Throwable,String)

    /**
     * Log an exception.
     *
     * @param   level   The level of the message.
     * @param   clazz   The name of the sending class.
     * @param   method  The name of the sending method.
     * @param   throwable The Throwable.
     * @param   detail  Some extra info to log with the Exception name.
     */
    public static void log(int level, String clazz, String method,
                           Throwable throwable, String detail)
    {
        long now = System.currentTimeMillis();

        // log the name and the message
        log(level, now, clazz, method,
            throwable.getClass().getName() +
            ": " + throwable.getMessage() +
            " - " + detail);


        // log the stacktrace
        final String stacktrace = getStackTrace(throwable);
        log(level + 1, now, clazz, method, stacktrace);
    }

    // }}}

    //----------------------------------------------------------------------
    // Output writer
    //----------------------------------------------------------------------
    // {{{ getLogWriter

    /**
     * Get the {@link LogWriter LogWriter} used for the default logging
     * mechanism, i.e. to log the messages that have passed through the filter
     * chain. <p>
     *
     * @return The LogWriter currently in use.
     *
     * @since   0.98.0
     */
    public static LogWriter getLogWriter()
    {
        return writer;
    }

    // }}}
    // {{{ setLogWriter

    /**
     * Set the {@link LogWriter LogWriter} to use for the default logging
     * mechanism, i.e. to log the messages that have passed through the filter
     * chain. <p>
     *
     * The default writer is a {@link StreamLogWriter StreamLogWriter} that
     * writes the messages to System.err using a {@link DefaultLogFormat
     * DefaultLogFormat}. <p>
     *
     * Setting the writer to null will suppress the output.
     *
     * @param logWriter The LogWriter to use.
     *
     * @since   0.98.0
     */
    public static void setLogWriter(LogWriter logWriter)
    {
        writer = logWriter;
    }

    // }}}

    //----------------------------------------------------------------------
    // Various filters and levels
    //----------------------------------------------------------------------
    // {{{ excludeThread

    /**
     * Exclude the current Thread from logging i.e. ignore every call to
     * {@link #log log} from the current Thread.
     */
    public static void excludeThread()
    {
        excludeThread(Thread.currentThread());
    }

    /**
     * Exclude a Thread from logging i.e. ignore every call to
     * {@link #log log} from the  Thread.
     *
     * @param   thread  The Thread to exclude.
     */
    public static void excludeThread(Thread thread)
    {
        synchronized (silenced) {
            // Thread may be excluded more than once, count the numbers
            if (silenced.containsKey(thread)) {
                silenced.put(thread,
                             Integer.valueOf(((Integer) silenced.get(thread))
                                          .intValue() + 1));
            } else {
                silenced.put(thread, Integer.valueOf(1));
            }
        }
    }

    // }}}
    // {{{ includeThread

    /**
     * Let the current Thread log again.
     */
    public static void includeThread()
    {
        includeThread(Thread.currentThread());
    }

    /**
     * Let a Thread log again.
     *
     * @param   thread  The Thread to log entries from.
     */
    public static void includeThread(Thread thread)
    {
        synchronized (silenced) {
            Integer old = (Integer) silenced.get(thread);
            if (old != null) {
                if (old.intValue() == 1) {
                    silenced.remove(thread);
                } else {
                    silenced.put(thread,
                                 Integer.valueOf(old.intValue() - 1));
                }
            }
        }
    }

    // }}}
    // {{{ addFilter

    /**
     * Add a LogFilter to the front of the filter chain.
     *
     * @param   filter  The filter to add.
     */
    public static void addFilter(LogFilter filter)
    {
        filters.insertElementAt(filter, 0);
        _filters = new LogFilter[filters.size()];
        filters.copyInto(_filters);
    }

    // }}}
    // {{{ removeFilter

    /**
     * Remove a LogFilter from the filter chain.
     *
     * @param   filter  The filter to remove
     */
    public static void removeFilter(LogFilter filter)
    {
        if (filters.removeElement(filter)) {
            _filters = new LogFilter[filters.size()];
            filters.copyInto(_filters);
        }
    }

    // }}}
    // {{{ getOutputLevel

    /**
     * Get the current output log level.
     *
     * @return  The current output log level.
     */
    public static int getOutputLevel()
    {
        return level;
    }

    // }}}
    // {{{ setOutputLevel

    /**
     * Set the current output log level. <p>
     *
     * After passing through both filter chains and the pre-queue level test,
     * messages will be checked against this value. If their level is equal to
     * or less than the current outout log level, thy will be printed to
     * System.err.
     *
     * @param level     Output log level to set.
     */
    public static void setOutputLevel(int level)
    {
        Log.level = level;
    }

    // }}}
    // {{{ getPreQueueLevel

    /**
     * Return the current pre-queue level.
     *
     * @return  The current pre-queue level.
     */
    public static int getPreQueueLevel()
    {
        return preQueueLevel;
    }

    // }}}
    // {{{ setPreQueueLevel

    /**
     * Set the current pre-queue level. <p>
     *
     * Messages that have passed through the pre-queue filter chain will have
     * their level checked against this value. If it is greater, they will be
     * discarded, otherwise they will be passed on, either to be queued or sent
     * through the post-queue filter chain directly.
     *
     * @param level     The pre-queue level to set.
     */
    public static void setPreQueueLevel(int level)
    {
        preQueueLevel = level;
    }

    // }}}
    // {{{ getMaxMessageSize

    /**
     * Get the maximum message size.
     *
     * @return  The maximum message size.
     */
    public static int getMaxMessageSize()
    {
        return maxMessageSize;
    }

    // }}}
    // {{{ setMaxMessageSize

    /**
     * Set the maximum message size.
     *
     * @param   size  The size to set.
     */
    public static void setMaxMessageSize(int size)
    {
        maxMessageSize = size;
    }

    // }}}

    //----------------------------------------------------------------------
    // The flush buffer
    //----------------------------------------------------------------------
    // {{{ getFlushBufferSize

    /**
     * Get the size of the flush buffer.
     *
     * @return  The flush buffer size or 0 to indicate no flush buffer.
     */
    public static int getFlushBufferSize()
    {
        synchronized(classlock) {
            return flushBuffer == null ? 0 : flushBuffer.getSize();
        }
    }

    // }}}
    // {{{ setFlushBufferSize

    /**
     * Set the size of the flush buffer. Setting it to 0 will turn the flush
     * buffer off.
     *
     * @param   size    The size of the flush buffer.
     */
    public static void setFlushBufferSize(int size)
    {
        synchronized(classlock) {
            if (size > 0) {
                if (flushBuffer == null) {
                    flushBuffer = new FlushBuffer(size);
                } else {
                    flushBuffer.resize(size);
                }
            } else {
                flushBuffer = null;
            }
        }
    }

    // }}}
    // {{{ getFlushTriggerLevel

    /**
     * Get the level that will trigger a flush of the messages saved in the
     * flush buffer.
     *
     * @return  The flush buffer's trigger level.
     */
    public static int getFlushTriggerLevel()
    {
        return flushTriggerLevel;
    }

    // }}}
    // {{{ setFlushTriggerLevel

    /**
     * Set the level that will trigger a flush of the messages saved in the
     * flush buffer.
     *
     * @param   level   The trigger level to set.
     */
    public static void setFlushTriggerLevel(int level)
    {
        flushTriggerLevel = level;
    }

    // }}}
    // {{{ getPostFlushSize

    /**
     * Get the number of messages to pass unfiltered through the pre-queue
     * stage after a flush happened.
     *
     * @return  The number of messages to pass.
     */
    public static int getPostFlushSize()
    {
        return postFlushSize;
    }

    // }}}
    // {{{ setPostFlushSize

    /**
     * Set the number of messages to pass unfiltered through the pre-queue
     * stage after a flush happened.
     *
     * @param   size    The number of messages to pass.
     */
    public static void setPostFlushSize(int size)
    {
        postFlushSize = size;
        if (postFlushCount > postFlushSize) {
            postFlushCount = postFlushSize;
        }
    }

    // }}}

    //----------------------------------------------------------------------
    // queueing
    //----------------------------------------------------------------------
    // {{{ isQueueing

    /**
     * Get the queueing state of the Log.
     *
     * @return  The queueing state of the Log.
     */
    public final static boolean isQueueing()
    {
        return queueing;
    }

    // }}}
    // {{{ setQueueing

    /**
     * Set the queueing state of the Log. If queueing is true, log entries
     * will be put on a queue and read by a low priority Thread during
     * idle time. In this way, logging has the least performance impact, but
     * entries logged just before a call to System.exit probably will not
     * be logged.<p>
     *
     * Setting the queueing flag to false will cause the queue to be flushed.
     *
     * @param   _queueing        The queueing state to set.
     */
    public final static void setQueueing(boolean _queueing)
    {
        queue.setQueueing(_queueing);
        queueing = _queueing;
    }

    // }}}
    // {{{ getQueueSize

    /**
     * Get the size of the queue where it starts to either block or drop
     * entries.
     *
     * @return  The maximum size of the queue.
     */
    public static int getQueueSize()
    {
        synchronized (logThread) {
            if (logThread.isRunning()) {
                return queue.getQueueSize();
            } else {
                return targetQueueSize;
            }
        }
    }

    // }}}
    // {{{ setQueueSize

    /**
     * Set the maximum size of the log queue. <p>
     *
     * When the log queue is full it will either block until entries are
     * processed ore start to drop old entries, as determined by {@link
     * #setDropOnOverflow setDropOnOverflow}. <p>
     *
     * The default maximum size is 3000.
     *
     * @param   size    The new size of the queue.
     */
    public static void setQueueSize(int size)
    {
        synchronized (logThread) {
            if (logThread.isRunning()) {
                queue.setQueueSize(size);
            } else {
                targetQueueSize = size;
            }
        }
    }


    // }}}
    // {{{ isDropOnOverflow

    /**
     * Query whether the queue drops entries when it overflows.
     *
     * @return  True if the queue drops entries on overflow.
     */
    public static boolean isDropOnOverflow()
    {
        synchronized (logThread) {
            if (logThread.isRunning()) {
                return queue.isDropOnOverflow();
            } else {
                return targetQueueDropOnOverflow;
            }
        }
    }

    // }}}
    // {{{ setDropOnOverflow

    /**
     * Set whether entries should be dropped when the log queue overflows. If
     * this is set to false, as is the default, the queue will block when it
     * is full, otherwise it will silently throw away old entries. <p>
     *
     * This setting should be used with care, especially when logging to a
     * file, since missing log entries can be misleading. It is useful during
     * debugging though, when only the last few messages are of interest
     * anyway.
     *
     * @param   drop    Whether entries should be dropped.
     */
    public static void setDropOnOverflow(boolean drop)
    {
        synchronized (logThread) {
            if (logThread.isRunning()) {
                queue.setDropOnOverflow(drop);
            } else {
                targetQueueDropOnOverflow = drop;
            }
        }
    }

    // }}}

    //----------------------------------------------------------------------
    // internals
    //----------------------------------------------------------------------
    // {{{ filter

    /**
     * Pass a LogEntry through the filter chain.
     *
     * @param   le      The LogEntry to filter.
     */
    private static void filter(final LogEntry le)
    {
        java.security.AccessController.doPrivileged(new java.security.PrivilegedAction () {
            @Override
            public Object run() {
                try {
                    boolean skip = false;
                    for (int i = 0; i < _filters.length; i++) {
                        if (! _filters[i].filter(le)) {
                            skip = true;
                            break;
                        }
                    }
                    if (writer != null && !skip && le.getLevel() <= level) {
                        writer.write(le);
                    }
                } catch (Exception e) {
                    if (DEBUG) {
                        System.err.println("Exception in LogThread"); // checkcode:ignore
                        e.printStackTrace(); //checkcode:ignore
                        System.err.println(le); // checkcode:ignore
                    }
                }
                return null;
            }
        });
    }

    /**
     * Pass an array of LogEntries through the filter chain.
     *
     * @param   entries      The LogEntries to filter.
     */
    private static void filter(final LogEntry[] fentries)
    {
        java.security.AccessController.doPrivileged(new java.security.PrivilegedAction () {
            @Override
            public Object run() {
                LogEntry[] entries = fentries;
                try {
                    for (int i = 0;
                         i < _filters.length && entries != null && entries.length > 0;
                         i++) {
                        entries = _filters[i].filter(entries);
                    }
                    if (writer != null && entries != null) {
                        for (int i = 0; i < entries.length; i++) {
                            if (entries[i].getLevel() <= level) {
                                writer.write(entries[i]);
                            }
                        }
                    }
                } catch (Exception e) {
                    if (DEBUG) {
                        System.err.println("Exception in LogThread"); // checkcode:ignore
                        e.printStackTrace(); //checkcode:ignore
                    }
                }
                return null;
            }
        });
    }

    // }}}
    // {{{ class LogThread

    /**
     * Helper class that takes Log messages from the queue and runs
     * them through the filter chain.
     */
    private static class LogThread extends Thread
    {
        private boolean running;

        /**
         * Create a new LogThread.
         */
        public LogThread()
        {
            setPriority(Thread.MIN_PRIORITY);
            setName("LogThread");
            setDaemon(true);
        }

        /**
         * Get all available LogEntries from the LogQueue and pass them
         * through the filter chain.
         */
        @Override
        public void run() {
            java.security.AccessController.doPrivileged(new java.security.PrivilegedAction () {
                @Override
                public Object run() {
                    LogEntry[] entries;
                    synchronized (this) {
                        running = true;
                        if (queueing) {
                            entries = queue.popAll();
                        } else {
                            entries = new LogEntry[] {queue.pop()};
                        }
                        queue.entriesProcessed();
                        queue.setQueueSize(targetQueueSize);
                        queue.setDropOnOverflow(targetQueueDropOnOverflow);
                    }
                    filter(entries);
                    while (true) {
                        if (queueing) {
                            filter(queue.popAll());
                        } else {
                            filter(queue.pop());
                        }
                        queue.entriesProcessed();
                    }
                }
            });
        }

        public final boolean isRunning()
        {
            return running;
        }
    }

    // }}}
    // {{{ getStackTrace

    /**
     * Get the stack trace of a throwable. Helper method for displaying and logging stacktraces.
     *
     * @param   ex      The Throwable.
     *
     * @return  The stack trace.
     */
    public static String getStackTrace(final Throwable ex)
    {
        final StringWriter os = new StringWriter();
        final PrintWriter pw = new PrintWriter (os);
        ex.printStackTrace(pw);
        pw.flush();
        return os.toString();
    }

    // }}}
    // {{{ class DumpStackException

    /**
     * Helper class to mark an exception which is only used to dump a log stack
     *
     */
    public static class DumpStackException extends Exception {
        public DumpStackException(final String message)
        {
            super(message);
        }

        @Override
        public Throwable fillInStackTrace() {
            final Throwable result = super.fillInStackTrace();
            cleanStackTrace();
            return result;

        }

        private void cleanStackTrace() {

            // remove all "dumpStack"-methods from the beginning of the stacktrace
            try {
                final StackTraceElement[] stackTraceElements = getStackTrace();
                int i = 0;
                for (final StackTraceElement element: stackTraceElements) {
                    final String methodName = element.getMethodName();
                    final boolean isInDumpStackMethod = "dumpStack".equals(methodName)
                            || "log".equals(methodName)
                            || "logAt".equals(methodName);
                    if (! isInDumpStackMethod) {
                        break;
                    }
                    i++;
                }
                if (i > 0) {
                    int length = stackTraceElements.length;

                    final StackTraceElement[] newElements = new StackTraceElement[length - i];
                    System.arraycopy(stackTraceElements,i,newElements,0,length - i);
                    setStackTrace(newElements);
                }
            } catch (final Throwable internEx) {
                // ignore, just use full stack trace then
            }
        }


    }
    // }}}
}
