// {{{ copyright

/********************************************************************
 *
 * The contents of this file are subject to the Mozilla Public
 * License Version 1.1 (the "License"); you may not use this file
 * except in compliance with the License. You may obtain a copy of
 * the License at http://www.mozilla.org/MPL/
 *
 * Software distributed under the License is distributed on an "AS
 * IS" basis, WITHOUT WARRANTY OF ANY KIND, either express or
 * implied. See the License for the specific language governing
 * rights and limitations under the License.
 *
 * The Original Code is qfs.de code.
 *
 * The Initial Developer of the Original Code is Gregor Schmid.
 * Portions created by Gregor Schmid are
 * Copyright (C) 1999 Quality First Software, Gregor Schmid.
 * All Rights Reserved.
 *
 * Contributor(s):
 *
 *******************************************************************/

// }}}

package de.qfs.lib.log;

// {{{ imports

import java.io.Serializable;

import java.text.DateFormat;
import java.text.SimpleDateFormat;

import java.util.Date;

// }}}

/**
 * A LogEntry is a container for the elements of a log message.
 *
 * @author      Gregor Schmid
 */
public class LogEntry implements Serializable
{
    // {{{ variables

    /**
     * The UID for serialization.
     */
    final static long serialVersionUID = -5697107084092473408L;

    /**
     * Helper to print timestamps.
     */
    private static Date date = new Date();

    /**
     * StringBuffer used for message output.
     */
    private static StringBuffer buffer = new StringBuffer ();

    /**
     * Helper to print timestamps.
     */
    private static SimpleDateFormat df = new SimpleDateFormat("HH:mm:ss.SSS");

    /**
     * The level at which the message was logged.
     */
    private int level;

    /**
     * The timestamp for the log message.
     */
    private long timestamp;

    /**
     * The name of the Thread that logged the message.
     */
    private String thread;

    /**
     * The name of the Class that logged the message.
     */
    private String clazz;

    /**
     * The method that logged the message.
     */
    private String method;

    /**
     * The message itself.
     */
    private String message;

    // }}}

    // {{{ constructor

    /**
     * Construct a new LogEntry.
     *
     * @param   level   The log level of the entry.
     * @param   timestamp The time of the entry in milliseconds.
     * @param   thread  The thread that sent the message.
     * @param   clazz   The class of the sender of the message.
     * @param   method  The method that sent the message.
     * @param   message The message itself.
     */
    public LogEntry (int level, long timestamp, String thread,
                     String clazz, String method, String message)
    {
        this.level = level;
        this.timestamp = timestamp;
        this.thread = thread;
        this.clazz = clazz;
        this.method = method;
        this.message = message;
    }

    // }}}

    // {{{ getLevel

    /**
     * @return  The level of the message.
     */
    public final int getLevel ()
    {
        return level;
    }

    // }}}
    // {{{ setLevel

    /**
     * Set the level of the message.
     *
     * @param   level   The level to set.
     */
    public final void setLevel(int level)
    {
        this.level = level;
    }

    // }}}
    // {{{ getThread

    /**
     * @return  The originating thread.
     */
    public final String getThread()
    {
        return thread;
    }

    // }}}
    // {{{ setThread

    /**
     * Set the thread of the message.
     *
     * @param   thread  The thread to set.
     */
    public final void setThread(String thread)
    {
        this.thread = thread;
    }

    // }}}
    // {{{ getTimestamp

    /**
     * @return  The time the message was sent (from currentTimeMillis).
     */
    public final long getTimestamp ()
    {
        return timestamp;
    }

    // }}}
    // {{{ setTimestamp

    /**
     * Set the timestamp of the message.
     *
     * @param   timestamp       The timestamp to set.
     */
    public final void setTimestamp(long timestamp)
    {
        this.timestamp = timestamp;
    }

    // }}}
    // {{{ getClazz

    /**
     * @return  The name of the class that sent the message.
     */
    public final String getClazz ()
    {
        return clazz;
    }

    // }}}
    // {{{ setClazz

    /**
     * Set the clazz of the message.
     *
     * @param   clazz   The clazz to set.
     */
    public final void setClazz(String clazz)
    {
        this.clazz = clazz;
    }

    // }}}
    // {{{ getMethod

    /**
     * @return  The method that sent the message.
     */
    public final String getMethod ()
    {
        return method;
    }

    // }}}
    // {{{ setMethod

    /**
     * Set the method of the message.
     *
     * @param   method  The method to set.
     */
    public final void setMethod(String method)
    {
        this.method = method;
    }

    // }}}
    // {{{ getMessage

    /**
     * @return  The text of the message.
     */
    public final String getMessage ()
    {
        return message;
    }

    // }}}
    // {{{ setMessage

    /**
     * Set the message.
     *
     * @param   message The message to set.
     */
    public final void setMessage(String message)
    {
        this.message = message;
    }

    // }}}

    // {{{ toString

    /**
     * Override the toString method to return a formatted String containing
     * all the relevant data of the message.
     *
     * @return  The formatted message
     */
    @Override
    public String toString()
    {
        synchronized(date) {
            date.setTime(timestamp);
            buffer.setLength(0);
            buffer.append(level);
            buffer.append(" (");
            buffer.append(df.format(date));
            buffer.append(") ");
            buffer.append(thread);
            buffer.append(" ");
            buffer.append(clazz);
            buffer.append(".");
            buffer.append(method);
            buffer.append(": ");
            buffer.append(message);
            return buffer.toString();
        }
    }

    // }}}
    // {{{ toString

    /**
     * Print the LogEntry using a DateFormat for the timestamp.
     *
     * @param   format  The DateFormat to use.
     *
     * @return  The formatted message
     *
     * @deprecated Please use the {@link LogFormat LogFormat} interface to
     * customize LogEntry output.
     */
    public String toString(DateFormat format)
    {
        // synchronization also protects buffer
        synchronized(date) {
            date.setTime(timestamp);
            buffer.setLength(0);
            buffer.append(level);
            buffer.append(" (");
            buffer.append(format.format(date));
            buffer.append(") ");
            buffer.append(thread);
            buffer.append(" ");
            buffer.append(clazz);
            buffer.append(".");
            buffer.append(method);
            buffer.append(": ");
            buffer.append(message);
            return buffer.toString();
            //  return ""  + level + " (" +
                //  format.format(date) + ") " +
                //  thread + " " + clazz + "." + method + ": " + message;
        }
    }

    // }}}
}
