// {{{ copyright

/********************************************************************
 *
 * The contents of this file are subject to the Mozilla Public
 * License Version 1.1 (the "License"); you may not use this file
 * except in compliance with the License. You may obtain a copy of
 * the License at http://www.mozilla.org/MPL/
 *
 * Software distributed under the License is distributed on an "AS
 * IS" basis, WITHOUT WARRANTY OF ANY KIND, either express or
 * implied. See the License for the specific language governing
 * rights and limitations under the License.
 *
 * The Original Code is qfs.de code.
 *
 * The Initial Developer of the Original Code is Gregor Schmid.
 * Portions created by Gregor Schmid are
 * Copyright (C) 1999 Quality First Software, Gregor Schmid.
 * All Rights Reserved.
 *
 * Contributor(s):
 *
 *******************************************************************/

// }}}

package de.qfs.lib.log;

/**
 * Callback that enables access to the log levels of an application as well as
 * the ouput and pre queue levels and the queue and flush buffer parameters.
 *
 * @author      Gregor Schmid
 * @see         Log
 */
public interface LogLevelCallback
{
    /**
     * Get the current log levels from the callback. This method is needed to
     * synchronize the LogLevelListener and the LogLevelCallback after the
     * connection has been established.
     *
     * @return  An object array that contains an alternating sequence of
     *          class/package names and log levels. The levels are Integers
     *          that may be null for classes for which a level has been
     *          requested but not explicitly set.
     */
    public Object[] getLogLevels();

    /**
     * Callback method for a LogLevelListener to change the log level for
     * a class or package.
     *
     * @param   name    The name of the affected class or package.
     * @param   level   The new log level.
     */
    public void setLogLevel(String name, int level);

    /**
     * Callback method for a LogLevelListener to remove the log level for
     * a class or package.
     *
     * @param   name    The name of the affected class or package.
     */
    public void removeLogLevel(String name);

    /**
     * Get the current output log level from the callback's application.
     *
     * @return  The current output log level.
     *
     * @see     Log#getOutputLevel
     */
    public int getOutputLevel();

    /**
     * Set the output log level for the callback's application.
     *
     * @param level     Output log level to set.
     *
     * @see     Log#setOutputLevel
     */
    public void setOutputLevel(int level);

    /**
     * Return the current pre-queue level from the callback's application.
     *
     * @return  The current pre-queue level.
     *
     * @see     Log#getPreQueueLevel
     */
    public int getPreQueueLevel();

    /**
     * Set the pre-queue level for the callback's application.
     *
     * @param level     The pre-queue level to set.
     *
     * @see     Log#setPreQueueLevel
     */
    public void setPreQueueLevel(int level);

    /**
     * Check whether the callback's application is using a log queue.
     *
     * @return  Whether the callback's application is queueing log entries.
     *
     * @see     Log#isQueueing
     */
    public boolean isQueueing();

    /**
     * Set whether the callback's application should use a log queue.
     *
     * @param   queue   Whether to queue log entries.
     *
     * @see     Log#setQueueing
     */
    public void setQueueing(boolean queue);

    /**
     * Get the size of log queue of the callback's application.
     *
     * @return  The size of the log queue.
     *
     * @see     Log#getQueueSize
     */
    public int getQueueSize();

    /**
     * Set the size of log queue of the callback's application.
     *
     * @param   size    The size of the log queue.
     *
     * @see     Log#setQueueSize
     */
    public void setQueueSize(int size);

    /**
     * Check whether the callback's application is dropping entries when
     * its log queue overflows.
     *
     * @return  Whether entries are dropped on overflow.
     *
     * @see     Log#isDropOnOverflow
     */
    public boolean isDropOnOverflow();

    /**
     * Set whether the callback's application should drop entries when
     * its log queue overflows.
     *
     * @param   drop    Whether to drop entries on overflow.
     *
     * @see     Log#setDropOnOverflow
     */
    public void setDropOnOverflow(boolean drop);

    /**
     * Get the size of the flush buffer of the callback's application.
     *
     * @return  The flush buffer size or 0 to indicate no flush buffer.
     *
     * @see     Log#getFlushBufferSize
     */
    public int getFlushBufferSize();

    /**
     * Set the size of the flush buffer of the callback's application. Setting
     * it to 0 will turn the flush buffer off.
     *
     * @param   size    The size of the flush buffer.
     *
     * @see     Log#setFlushBufferSize
     */
    public void setFlushBufferSize(int size);

    /**
     * Get the level that will trigger a flush of the messages saved in the
     * flush buffer of the callback's application.
     *
     * @return  The flush buffer's trigger level.
     *
     * @see     Log#getFlushTriggerLevel
     */
    public int getFlushTriggerLevel();

    /**
     * Set the level that will trigger a flush of the messages saved in the
     * flush buffer of the callback's application.
     *
     * @param   level   The trigger level to set.
     *
     * @see     Log#setFlushTriggerLevel
     */
    public void setFlushTriggerLevel(int level);

    /**
     * Get the number of messages to pass unfiltered through the pre-queue
     * stage after a flush happened in the callback's application.
     *
     * @return  The number of messages to pass.
     *
     * @see     Log#getPostFlushSize
     */
    public int getPostFlushSize();

    /**
     * Set the number of messages to pass unfiltered through the pre-queue
     * stage after a flush happened in the callback's application.
     *
     * @param   size    The number of messages to pass.
     *
     * @see     Log#setPostFlushSize
     */
    public void setPostFlushSize(int size);
}
