// {{{ copyright

/********************************************************************
 *
 * The contents of this file are subject to the Mozilla Public
 * License Version 1.1 (the "License"); you may not use this file
 * except in compliance with the License. You may obtain a copy of
 * the License at http://www.mozilla.org/MPL/
 *
 * Software distributed under the License is distributed on an "AS
 * IS" basis, WITHOUT WARRANTY OF ANY KIND, either express or
 * implied. See the License for the specific language governing
 * rights and limitations under the License.
 *
 * The Original Code is qfs.de code.
 *
 * The Initial Developer of the Original Code is Gregor Schmid.
 * Portions created by Gregor Schmid are
 * Copyright (C) 1999 Quality First Software, Gregor Schmid.
 * All Rights Reserved.
 *
 * Contributor(s):
 *
 *******************************************************************/

// }}}

package de.qfs.lib.log;

// {{{ imports

import java.util.EventObject;

// }}}

/**
 * LogLevelEvent is used to notify {@link LogLevelListener LogLevelListeners}
 * of changes in the log levels used by the {@link Logger Loggers} of an
 * application.
 *
 * @author      Gregor Schmid
 */
public class LogLevelEvent
    extends EventObject
{
    // {{{ variables

    /**
     * The name of the affected class or package.
     *
     * @serial
     */
    private String[] names;

    /**
     * A (possibly) new log level.
     *
     * @serial
     */
    private int[] levels;

    // }}}

    // {{{ constructors

    /**
     * Create a new LogLevelEvent.
     *
     * @param   source  The source of the event.
     * @param   name    The name of the affected class or package.
     */
    public LogLevelEvent(Object source, String name)
    {
        super(source);
        names = new String[] {name};
    }

    /**
     * Create a new LogLevelEvent.
     *
     * @param   source  The source of the event.
     * @param   name    The name of the affected class or package.
     * @param   level   The new log level.
     */
    public LogLevelEvent(Object source, String name, int level)
    {
        super(source);
        names = new String[] {name};
        levels = new int[] {level};
    }

    /**
     * Create a new LogLevelEvent.
     *
     * @param   source  The source of the event.
     * @param   names   The names of the affected classes or packages.
     */
    public LogLevelEvent(Object source, String[] names)
    {
        super(source);
        this.names = names;
    }

    // }}}

    // {{{ getNames

    /**
     * Get the names of the affected classes or packages.
     *
     * @return  The names.
     */
    public final String[] getNames()
    {
        return names;
    }

    // }}}
    // {{{ getLevels

    /**
     * Get the new log levels.
     *
     * @return  The levels.
     */
    public final int[] getLevels()
    {
        return levels;
    }

    // }}}
}
