// {{{ copyright

/********************************************************************
 *
 * The contents of this file are subject to the Mozilla Public
 * License Version 1.1 (the "License"); you may not use this file
 * except in compliance with the License. You may obtain a copy of
 * the License at http://www.mozilla.org/MPL/
 *
 * Software distributed under the License is distributed on an "AS
 * IS" basis, WITHOUT WARRANTY OF ANY KIND, either express or
 * implied. See the License for the specific language governing
 * rights and limitations under the License.
 *
 * The Original Code is qfs.de code.
 *
 * The Initial Developer of the Original Code is Gregor Schmid.
 * Portions created by Gregor Schmid are
 * Copyright (C) 1999 Quality First Software, Gregor Schmid.
 * All Rights Reserved.
 *
 * Contributor(s):
 *
 *******************************************************************/

// }}}

package de.qfs.lib.log;

// {{{ imports

import java.util.Enumeration;
import java.util.Hashtable;
import java.util.Vector;

// }}}

/**
 * This class manages log levels for Classes and packages in a tree
 * hierarchy. Loggers requesting their log level will be registered so they
 * can be notified of changes. <p>
 *
 * {@link LogLevelListener LogLevelListeners} can be added to monitor changes
 * in the log levels. These changes will be propagated by a background thread
 * to avoid delays and problems in the main application.
 *
 * @author      Gregor Schmid
 */
public class LogLevels
{
    // {{{ variables

    /**
     * The root of the log level tree.
     */
    Node root = new Node ("", null,
                                  Integer.valueOf(Logger.DEFAULT_LOG_LEVEL));

    /**
     * The list of LogLevelListeners.
     */
    private Vector listeners = new Vector ();

    /**
     * The list of LogLevelListeners as an array to avoid unnecessary
     * memory consumption.
     */
    private LogLevelListener[] _listeners = new LogLevelListener[0];

    /**
     * The background thread that serves the listeners.
     */
    private LogLevelThread logLevelThread = new LogLevelThread ();

    // }}}

    // {{{ constructor

    /**
     * Create a new LogLevels object.
     */
    public LogLevels ()
    {
    }

    // }}}

    //----------------------------------------------------------------------
    // Changing levels
    //----------------------------------------------------------------------
    // {{{ setDefaultLogLevel

    /**
     * Set the default log level for all classes, for which no level is set
     * either directly on the class or on (parts of) its package. This not
     * only affects future calls to {@link #getLogLevel getLogLevel}, but also
     * changes the log level of already registered loggers.
     *
     * @param   level   The new default level.
     */
    public void setDefaultLogLevel(int level)
    {
        setDefaultLogLevelImpl(this, level);
    }

    // }}}
    // {{{ setLogLevel

    /**
     * Set the log level for a class or package. If the name represents a
     * package or part of a package it must end with '.' to distinguish it
     * from a class name. Otherwise setting the level for e.g.
     * "de.qfs.lib.log.Log" would implicitly set the level for
     * "de.qfs.lib.log.Logger" as well.
     *
     * @param   name    The name of the class or package (part). Package names
     *                  must end with a dot ('.').
     * @param   level   The maximum level at which objects of the class or
     *                  classes from the package should log messages.
     */
    public void setLogLevel(String name, int level)
    {
        if (name.length() == 0) {
            setDefaultLogLevelImpl(this, level);
        } else {
            setLogLevelImpl(this, name, level);
        }
    }

    // }}}
    // {{{ removeLogLevel

    /**
     * Remove the level that was set for a class or package so the
     * class or the package's classes will now inherit the default log level
     * or a package (part) level.
     *
     * @param   name    The name of the class or package (part). Package names
     *                  must end with a dot.
     */
    public void removeLogLevel(String name)
    {
        if (name.length() == 0) {
            setDefaultLogLevelImpl(this, Logger.DEFAULT_LOG_LEVEL);
        } else {
            removeLogLevelImpl(this, name);
        }
    }

    // }}}

    //----------------------------------------------------------------------
    // Querying
    //----------------------------------------------------------------------
    // {{{ getDedicatedLogLevel

    /**
     * Get the log level for a Logger. Calling this method will <b>not<b/> register the
     * Logger requesting the level, nor return its parent level. Returns 0, if the level
     * has not been set.
     *
     * @param   loggerName  The name of the logger for which a level is requested
     *
     * @return The log level for the class owning the logger, if it has been
     *         explicitly set with {@link #setLogLevel setLogLevel}.
     */
    public synchronized int getDedicatedLogLevel(final String loggerName)
    {
        Node node = root.findNodeAndParent(loggerName)[0];

        return node == null || node.level == null ? 0 : node.level.intValue();
    }

    // }}}
    // {{{ getLogLevel

    /**
     * Get the log level for a Logger. Calling this method will register the
     * Logger requesting the level, so that later changes to the default
     * level, its package's level or explicitly for its class can change the
     * Logger's level.
     *
     * @param   logger  The Logger requesting its log level.
     *
     * @return The log level for the class owning the logger, if it has been
     *         explicitly set with {@link #setLogLevel setLogLevel}, the level
     *         of its package, or the default level set with {@link
     *         #setDefaultLogLevel setDefaultLogLevel} otherwise.
     */
    public synchronized int getLogLevel(Logger logger)
    {
        String name = logger.getOwnerName();

        Node[] nodes = root.findNodeAndParent(name);

        if (nodes[0] == null) {
            // Create a new Node for the class and add it to the parent. The
            // level for the child is set to null so it will inherit the
            // parent's level.
            nodes[0] = new Node (name, logger, null);
            nodes[1].addChild(nodes[0]);
            logLevelThread.classAdded(this, nodes[0].name);
        } else {
            // The child node is the exact node for the class.
            nodes[0].logger = logger;
        }
        // Return the child's level or the parent's if it is unset.
        return nodes[0].level != null
            ? nodes[0].level.intValue()
            : nodes[1].level.intValue();
    }

    // }}}

    //----------------------------------------------------------------------
    // Listeners
    //----------------------------------------------------------------------
    // {{{ addLogLevelListener

    /**
     * Add a LogLevelListener to the LogLevels.
     *
     * @param   listener        The listener to add.
     */
    public synchronized void addLogLevelListener(LogLevelListener listener)
    {
        if (listener == null) {
            Thread.dumpStack();
        }
        if (! listeners.contains(listener)) {
            listeners.addElement(listener);
            _listeners = new LogLevelListener[listeners.size()];
            listeners.copyInto(_listeners);
        }
    }

    // }}}
    // {{{ removeLogLevelListener

    /**
     * Remove a LogLevelListener from the LogLevels.
     *
     * @param   listener        The listener to remove.
     */
    public synchronized void removeLogLevelListener(LogLevelListener listener)
    {
        listeners.removeElement(listener);
        _listeners = new LogLevelListener[listeners.size()];
        listeners.copyInto(_listeners);
    }

    // }}}
    // {{{ fireClassAdded

    /**
     * Notify the listeners that a Node has been added.
     *
     * @param   event   The event containing the details.
     */
    private void fireClassAdded(LogLevelEvent event)
    {
        // Avoid concurrency problems
        LogLevelListener[] tmp;
        synchronized (this) {
            tmp = _listeners;
        }
        for (int i = 0; i < tmp.length; i++) {
            // must check with equals since different stubs may denote the
            // same RMI object
            if (! tmp[i].equals(event.getSource())) {
                //  try {
                    tmp[i].classAdded(event);
                //  } catch (RemoteException ex) {
                    // no mercy
                    //  removeLogLevelListener(tmp[i]);
                //  }
            }
        }
    }

    // }}}
    // {{{ fireLevelChanged

    /**
     * Notify the listeners of a change in a Node's level.
     *
     * @param   event   The event containing the details.
     */
    private void fireLevelChanged(LogLevelEvent event)
    {
        // Avoid concurrency problems
        LogLevelListener[] tmp;
        synchronized (this) {
            tmp = _listeners;
        }
        for (int i = 0; i < tmp.length; i++) {
            // must check with equals since different stubs may denote the
            // same RMI object
            if (! tmp[i].equals(event.getSource())) {
                //  try {
                    tmp[i].levelChanged(event);
                //  } catch (RemoteException ex) {
                    // no mercy
                    //  removeLogLevelListener(tmp[i]);
                //  }
            }
        }
    }

    // }}}
    // {{{ fireLevelRemoved

    /**
     * Notify the listeners that a Node has been removed.
     *
     * @param   event   The event containing the details.
     */
    private void fireLevelRemoved(LogLevelEvent event)
    {
        // Avoid concurrency problems
        LogLevelListener[] tmp;
        synchronized (this) {
            tmp = _listeners;
        }
        for (int i = 0; i < tmp.length; i++) {
            // must check with equals since different stubs may denote the
            // same RMI object
            if (! tmp[i].equals(event.getSource())) {
                //  try {
                    tmp[i].levelRemoved(event);
                //  } catch (RemoteException ex) {
                    // no mercy
                    //  removeLogLevelListener(tmp[i]);
                //  }
            }
        }
    }

    // }}}

    //----------------------------------------------------------------------
    // The LogLevelCallback interface
    //----------------------------------------------------------------------
    // {{{ getLogLevels

    /**
     * Get the current log levels from the callback. This method is needed to
     * synchronize the LogLevelListener and the LogLevelCallback after the
     * connection has been established.
     *
     * @return  An object array that contains an alternating sequence of
     *          class/package names and log levels. The levels are Integers
     *          that may be null for classes for which a level has been
     *          requested but not explicitly set.
     *
     * @throws  RemoteException If something RMI specific goes wrong.
     */
    public Object[] getLogLevels()
        //  throws RemoteException
    {
        Vector result = new Vector ();
        synchronized (this) {
            root.fillList(result);
        }
        Object[] ret = new Object [result.size()];
        result.copyInto(ret);
        return ret;
    }

    // }}}
    // {{{ setLogLevel

    /**
     * Callback method for a LogLevelListener to change the log level for
     * a class or package.
     *
     * @param   source  The listener that causes the change. It will not be
     *                  notified to avoid recursion.
     * @param   name    The name of the affected class or package.
     * @param   level   The new log level.
     *
     * @throws  RemoteException If something RMI specific goes wrong.
     */
    public void setLogLevel(LogLevelListener source, String name, int level)
        //  throws RemoteException
    {
        if (name.length() == 0) {
            setDefaultLogLevelImpl(source, level);
        } else {
            setLogLevelImpl(source, name, level);
        }
    }

    // }}}
    // {{{ removeLogLevel

    /**
     * Callback method for a LogLevelListener to remove the log level for
     * a class or package.
     *
     * @param   source  The listener that causes the change. It will not be
     *                  notified to avoid recursion.
     * @param   name    The name of the affected class or package.
     *
     * @throws  RemoteException If something RMI specific goes wrong.
     */
    public void removeLogLevel(LogLevelListener source, String name)
        //  throws RemoteException
    {
        // Default level can't be removed
        if (name.length() == 0) {
            setDefaultLogLevelImpl(source, Logger.DEFAULT_LOG_LEVEL);
        } else {
            removeLogLevelImpl(source, name);
        }
    }

    // }}}

    //----------------------------------------------------------------------
    // Helper methods
    //----------------------------------------------------------------------
    // {{{ setDefaultLogLevelImpl

    /**
     * Set the default log level for all classes, for which no level is set
     * either directly on the class or on (parts of) its package. This not
     * only affects future calls to {@link #getLogLevel getLogLevel}, but also
     * changes the log level of already registered loggers.
     *
     * @param   source  The source of the action.
     * @param   level   The new default level.
     */
    private synchronized void setDefaultLogLevelImpl(Object source, int level)
    {
        root.setLevelAndUpdate(level);
        logLevelThread.logLevelChanged(source, "", level);
    }

    // }}}
    // {{{ setLogLevelImpl

    /**
     * Set the log level for a class or package. If the name represents a
     * package or part of a package it must end with '.' to distinguish it
     * from a class name. Otherwise setting the level for e.g.
     * "de.qfs.lib.log.Log" would implicitly set the level for
     * "de.qfs.lib.log.Logger" as well.
     *
     * @param   source  The source of the action.
     * @param   name    The name of the class or package (part). Package names
     *                  must end with a dot ('.').
     * @param   level   The maximum level at which objects of the class or
     *                  classes from the package should log messages.
     */
    private synchronized void setLogLevelImpl(Object source, String name,
                                              int level)
    {
        Node[] nodes = root.findNodeAndParent(name);

        if (nodes[0] == null) {
            // Create a new Node for the class or package and add it to the
            // parent.
            nodes[0] = new Node (name, null, null);
            nodes[1].addChild(nodes[0]);
        }

        // Change the level for the Node and all of its direct Class Node
        // children.
        nodes[0].setLevelAndUpdate(level);
        logLevelThread.logLevelChanged(source, nodes[0].name, level);
    }

    // }}}
    // {{{ removeLogLevelImpl

    /**
     * Remove the level that was set for a class or package so the
     * class or the package's classes will now inherit the default log level
     * or a package (part) level.
     *
     * @param   source  The source of the action.
     * @param   name    The name of the class or package (part). Package names
     *                  must end with a dot.
     */
    private synchronized void removeLogLevelImpl(Object source, String name)
    {
        Node[] nodes = root.findNodeAndParent(name);

        if (nodes[0] == null) {
            // There is no such node, there is nothing to do.
            return;
        } else if (nodes[0].logger != null) {
            // This is a Class Node so it must be kept.
            nodes[0].unsetLevelAndUpdate(nodes[1].level.intValue());
        } else {
            // Remove the Node from its parent. This will automatically update
            // all the levels.
            nodes[1].removeChild(nodes[0]);
        }
        logLevelThread.logLevelRemoved(source, nodes[0].name);
    }

    // }}}

    //----------------------------------------------------------------------
    // The Node hierarchy
    //----------------------------------------------------------------------
    // {{{ class Node

    /**
     * The tree hierarchy for the LogLevels is built from Nodes. Each Node
     * represents either a class, a package or part of a package, as defined
     * by its name. Packages or parts thereof must end with a '.', Class names
     * mustn't, e.g. "de.qfs.lib.log.LogLevels", "de.qfs.lib.log." or
     * "de.qfs.". <p>
     *
     * The Logger member of the Node is set, when a Logger asks for its log
     * level via {@link #getLogLevel getLogLevel}. <p>
     *
     * The level member of the Node is set, when an explicit call to
     * {@link #setLogLevel setLogLevel} is made. <p>
     *
     * The Node hierarchy is optimized for lookup. <p>
     *
     * Node methods are not thread safe so synchronization must be provided at
     * a higher level.
     */
    class Node
    {
        // {{{ variables

        /**
         * The name of the Node. May be a class name, a package name or the
         * start of a package name. (Parts of) package names must end with a
         * dot.
         */
        public String name;

        /**
         * A Logger that requested its log level with {@link #getLogLevel
         * getLogLevel}.
         */
        public Logger logger;

        /**
         * The level for the Node, if it has been explicitly set with
         * {@link #setLogLevel setLogLevel}.
         */
        public Integer level;

        /**
         * The class Node children of the Node in no special order.
         */
        public Hashtable classes;

        /**
         * The package Node children of the Node in no special order.
         */
        public Vector packages;

        // }}}

        // {{{ constructor

        /**
         * Create a new Node.
         *
         * @param       name    The name of the Node.
         * @param       logger  The Logger of the Node.
         * @param       level   The level of the Node.
         */
        public Node(String name, Logger logger, Integer level)
        {
            this.name = name;
            this.logger = logger;
            this.level = level;
        }

        // }}}

        //------------------------------------------------------------------
        // The tree structure
        //------------------------------------------------------------------
        // {{{ findNodeAndParent

        /**
         * Find a Node and its parent, given the Node's name.
         *
         * @param       name    The name of the Node to look for.
         *
         * @return      A two element array. The first element is the sought
         *              Node, or null if no Node by that name exists. The
         *              second element is either the parent of the existing
         *              Node, or the Node in which a new Node by that name
         *              should be inserted.
         */
        public Node[] findNodeAndParent(String name)
        {
            if (classes != null) {
                Node child = (Node) classes.get(name);
                if (child != null) {
                    // This is the child we were looking for, so this Node is
                    // its parent.
                    return new Node[] {child, this};
                }
            }
            if (packages != null) {
                for (Enumeration i = packages.elements();
                     i.hasMoreElements(); ) {
                    Node child = (Node) i.nextElement();
                    if (name.equals(child.name)) {
                        // This is the child we were looking for, so this Node
                        // is its parent.
                        return new Node[] {child, this};
                    }
                    if (name.startsWith(child.name)) {
                        // The sought child must belong to that Node.
                        return child.findNodeAndParent(name);
                    }
                }
            }
            // The sought child does not exist but this Node would be the
            // correct parent.
            return new Node[] {null, this};
        }

        // }}}
        // {{{ addChild

        /**
         * Add a child to the Node. If the child represents a package (or part
         * of a package) it will inherit all of the current children of the
         * Node whose name starts with the package's name.
         *
         * @param       child   The new child.
         */
        public void addChild(Node child)
        {
            if (child.name.endsWith(".")) {
                // The new child is a package node that may adopt some of
                // this Node's children.

                // First check the classes.
                if (classes != null) {
                    // cannot modify classes while iterating
                    Vector tmp = new Vector ();
                    for (Enumeration i = classes.keys();
                         i.hasMoreElements(); ) {
                        String name = (String) i.nextElement();
                        if (name.startsWith(child.name)) {
                            // Transfer the child from the old to the new
                            // parent.
                            if (child.classes == null) {
                                child.classes = new Hashtable ();
                            }
                            child.classes.put(name, classes.get(name));
                            // remove later
                            tmp.addElement(name);
                        }
                    }
                    // now remove the transferred children
                    for (Enumeration i = tmp.elements();
                         i.hasMoreElements(); ) {
                        classes.remove(i.nextElement());
                    }
                }

                // And now the packages
                if (packages != null) {
                    for (int i = 0; i < packages.size(); i++) {
                        Node mychild = (Node) packages.elementAt(i);
                        if (mychild.name.startsWith(child.name)) {
                            // Transfer the child from the old to the new
                            // parent.
                            if (child.packages == null) {
                                child.packages = new Vector ();
                            }
                            packages.removeElementAt(i--);
                            child.packages.addElement(mychild);
                        }
                    }
                }

                // Now add the child to the packages
                if (packages == null) {
                    packages = new Vector ();
                }
                packages.addElement(child);
            } else {
                // just add the child to the classes
                if (classes == null) {
                    classes = new Hashtable ();
                }
                classes.put(child.name, child);
            }
        }

        // }}}
        // {{{ removeChild

        /**
         * Remove a child Node from the Node.
         *
         * @param       child   The Node to remove
         */
        public synchronized void removeChild(Node child)
        {
            if (child.name.endsWith(".")) {
                if (packages != null) {
                    packages.removeElement(child);
                    // Adopt the former grandchildren
                    if (child.classes != null) {
                        if (classes == null) {
                            classes = new Hashtable ();
                        }
                        for (Enumeration i = child.classes.elements();
                             i.hasMoreElements(); ) {
                            Node grandchild = (Node) i.nextElement();
                            if (grandchild.logger != null
                                && grandchild.level == null) {
                                // adoption may change the grandchild's
                                // inherited level
                                grandchild.logger.setLevel(level.intValue());
                            }
                            classes.put(grandchild.name, grandchild);
                        }
                    }
                    if (child.packages != null) {
                        for (Enumeration i = child.packages.elements();
                             i.hasMoreElements();) {
                            packages.addElement(i.nextElement());
                        }
                    }
                }
            } else if (classes != null) {
                // just remove it
                classes.remove(child.name);
            }
        }

        // }}}

        //------------------------------------------------------------------
        // Changing levels
        //------------------------------------------------------------------
        // {{{ setLevelAndUpdate

        /**
         * Set the level for a Node and update its Logger's level or the log
         * levels of its class Node children, if the Node is a package Node
         * and the child's level hasn't been explicitly set.
         *
         * @param       level   The level to set.
         */
        public void setLevelAndUpdate(int level)
        {
            this.level = Integer.valueOf(level);
            if (logger != null) {
                // This is a Class Node.
                logger.setLevel(level);
            } else if (classes != null) {
                // Look for Class Node children
                for (Enumeration i = classes.elements();
                     i.hasMoreElements(); ) {
                    Node child = (Node) i.nextElement();
                    if (child.level == null && child.logger != null) {
                        // This is a Class child Node for which no level has
                        // been explicitly set.
                        child.logger.setLevel(level);
                    }
                }
            }
        }

        // }}}
        // {{{ unsetLevelAndUpdate

        /**
         * Remove the explicitly set level for a Node and update its class'
         * log level. Only useful for Class Nodes.
         *
         * @param       level   The level to set
         */
        public void unsetLevelAndUpdate(int level)
        {
            if (this.level != null) {
                // Only in this case does anything change for the Class.
                this.level = null;
                if (logger != null) {
                    // This is a Class Node.
                    logger.setLevel(level);
                }
            }
        }

        // }}}

        //------------------------------------------------------------------
        // Walk the tree
        //------------------------------------------------------------------
        // {{{ fillList

        /**
         * Recursively add names and levels of all nodes to list.
         *
         * @param       list    The List to fill.
         */
        public void fillList(Vector list)
        {
            list.addElement(name);
            list.addElement(level);
            if (packages != null) {
                for (Enumeration i = packages.elements();
                     i.hasMoreElements(); ) {
                    ((Node) i.nextElement()).fillList(list);
                }
            }
            if (classes != null) {
                for (Enumeration i = classes.elements();
                     i.hasMoreElements(); ) {
                    ((Node) i.nextElement()).fillList(list);
                }
            }
        }

        // }}}
    }

    // }}}

    //----------------------------------------------------------------------
    // A background thread for event dispatch
    //----------------------------------------------------------------------
    // {{{ class LogLevelThread

    /**
     * Background thread that delivers LogLevelEvents to the LogLevelListeners.
     */
    private class LogLevelThread
        extends Thread
    {
        // {{{ variables

        /**
         * A Map from event sources to Lists of names of added classes.
         */
        private Hashtable added = new Hashtable ();

        /**
         * List of pending levelChanged events.
         */
        private Vector changed = new Vector ();

        /**
         * List of pending levelRemoved events.
         */
        private Vector removed = new Vector ();

        // }}}

        // {{{ constructor

        /**
         * Create a new LogLevelThread. Sets the priority to NORM_PRIORITY - 1
         * and automatically starts itself.
         */
        public LogLevelThread ()
        {
            super("LogLevelThread");
            setDaemon(true);
            setPriority(Thread.NORM_PRIORITY - 1);
            start();
        }

        // }}}
        // {{{ run

        /**
         * Wait for events to dispatch and forward them to the
         * LogLevelListeners.
         */
        @Override
        public void run()
        {
            java.security.AccessController.doPrivileged(new java.security.PrivilegedAction () {
                @Override
                public Object run() {
                    while (true) {
                        LogLevelEvent removedEvent = null;
                        LogLevelEvent changedEvent = null;
                        LogLevelEvent addedEvent = null;
                        synchronized (LogLevelThread.this) {
                            while (added.isEmpty()
                                   && changed.isEmpty()
                                   && removed.isEmpty()) {
                                try {
                                    LogLevelThread.this.wait();
                                } catch (InterruptedException ex) {
                                }
                            }
                            if (!removed.isEmpty()) {
                                removedEvent = (LogLevelEvent) removed.elementAt(0);
                                removed.removeElementAt(0);
                            } else if (!changed.isEmpty()) {
                                changedEvent = (LogLevelEvent) changed.elementAt(0);
                                changed.removeElementAt(0);
                            } else if (!added.isEmpty()) {
                                Object source = added.keys().nextElement();
                                Vector tmp = (Vector) added.remove(source);
                                String[] classes = new String[tmp.size()];
                                tmp.copyInto(classes);
                                addedEvent = new LogLevelEvent (source, classes);
                            }
                        }
                        if (removedEvent != null) {
                            fireLevelRemoved(removedEvent);
                            removedEvent = null;
                        } else if (changedEvent != null) {
                            fireLevelChanged(changedEvent);
                            changedEvent = null;
                        } else if (addedEvent != null) {
                            fireClassAdded(addedEvent);
                            addedEvent = null;
                        }
                    }
                }
            });
        }

        // }}}

        // {{{ classAdded

        /**
         * Tell the LogLevelThread that a class has been added.
         *
         * @param       source  The source of the event.
         * @param       name    The name of the added class.
         */
        public synchronized void classAdded(Object source, String name)
        {
            // System.err.println("added " + name);
            Vector tmp = (Vector) added.get(source);
            if (tmp == null) {
                tmp = new Vector ();
                added.put(source, tmp);
            }
            tmp.addElement(name);
            this.notifyAll();
        }

        // }}}
        // {{{ logLevelChanged

        /**
         * Tell the LogLevelThread that a log level has been changed.
         *
         * @param       source  The source of the event.
         * @param       name    The name of the affected class or package.
         * @param       level   The new log level.
         */
        public synchronized void logLevelChanged(Object source, String name,
                                                 int level)
        {
            //  System.err.println("changed " + name + " " + level);
            changed.addElement(new LogLevelEvent (source, name, level));
            this.notifyAll();
        }

        // }}}
        // {{{ logLevelRemoved

        /**
         * Tell the LogLevelThread that a log level has been removed.
         *
         * @param       source  The source of the event.
         * @param       name    The name of the affected class or package.
         */
        public synchronized void logLevelRemoved(Object source, String name)
        {
            //  System.err.println("removed " + name);
            removed.addElement(new LogLevelEvent (source, name));
            this.notifyAll();
        }

        // }}}
    }

    // }}}


}
