// {{{ copyright
// *******************************************************************
//
// Copyright (C) 2023 Quality First Software
// All rights reserved
//
// *******************************************************************
// }}}

package de.qfs.lib.log;

import lombok.AllArgsConstructor;
import lombok.NonNull;
import lombok.extern.qfs.QFLog;

/**
 * Helper Class to define specific object dumpers on loggers.
 *
 * By this, for a specific type of objects, an alternative logging
 * method can be defined instead of the default <code>toString()</code>.
 *
 */
@QFLog
@AllArgsConstructor
public class LogObjectDumper
{
    @NonNull final Class clazz;
    @NonNull final Method method;

    @FunctionalInterface
    public interface Method
    {
        /**
         * Adds an Object to the string builder.
         *
         * @return the StringBuilder if added, null if not.
         */
        StringBuilder add(StringBuilder sb, Object object);
    }

    public void addTo(final QFLogger logger) {
        if (logger == null) return;
        try {
            logger.addObjectDumper(this);
        } catch (final Throwable t) {
            // e.g. a MethodNotFound exception is thrown if QFLogger class is loaded from an old package where the addObjectDumper method is not supported, yet.
            qflog(WRN, "Could not register object dumper for class " + clazz.getName(), t);
        }
    }

    public static void register(final QFLogger logger, final Class clazz, final Method method) {
        if (logger == null || clazz == null || method == null) return;
        new LogObjectDumper(clazz, method).addTo(logger);
    }
}
