// {{{ copyright

/********************************************************************
 *
 * The contents of this file are subject to the Mozilla Public
 * License Version 1.1 (the "License"); you may not use this file
 * except in compliance with the License. You may obtain a copy of
 * the License at http://www.mozilla.org/MPL/
 *
 * Software distributed under the License is distributed on an "AS
 * IS" basis, WITHOUT WARRANTY OF ANY KIND, either express or
 * implied. See the License for the specific language governing
 * rights and limitations under the License.
 *
 * The Original Code is qfs.de code.
 *
 * The Initial Developer of the Original Code is Gregor Schmid.
 * Portions created by Gregor Schmid are
 * Copyright (C) 1999 Quality First Software, Gregor Schmid.
 * All Rights Reserved.
 *
 * Contributor(s):
 *
 *******************************************************************/

// }}}

package de.qfs.lib.log;

// {{{ imports


// }}}

/**
 * This is a queue specialized for LogEntries.
 *
 * @see Log
 * @author      Gregor Schmid
 */
public class LogQueue
{
    // {{{ variables

    /**
     * Whether we are really queueing or just act as a cubbyhole.
     */
    private boolean queueing = Log.DEFAULT_QUEUEING;

    /**
     * The cubbyhole object.
     */
    private LogEntry cubby;

    /**
     * The maximum size of the queue.
     */
    private int maxSize = Log.DEFAULT_INITIAL_QUEUE_SIZE;

    /**
     * Whether entries are dropped when the queue overflows.
     */
    private boolean drop = Log.DEFAULT_INITIAL_DROP_ON_OVERFLOW;

    /**
     * An internal flag used to signal the unit tests that the queue is going
     * to block until some entries have been popped.
     */
    boolean mustPop;

    /**
     * The array holding the entries.
     */
    private LogEntry[] array = new LogEntry[maxSize];

    /**
     * The head of the list.
     */
    private int head;

    /**
     * The tail of the list.
     */
    private int tail;

    /**
     * The number of entries in the list.
     */
    private int size = 0;

    /**
     * Whether to wait for flushing.
     */
    private boolean waitFlush;

    /**
     * The current flushCount. > 0 means entries are still being processed.
     */
    private int flushCount;

    // }}}

    //------------------------------------------------------------------------------------------
    // Constructor
    //------------------------------------------------------------------------------------------
    // {{{ LogQueue

    /**
     * Create a new LogQueue.
     */
    public LogQueue ()
    {
    }

    // }}}

    //------------------------------------------------------------------------------------------
    // Pushing and popping
    //------------------------------------------------------------------------------------------
    // {{{ push

    /**
     * Push an object to the back of the queue.<P>
     *
     * @param   object  The object being pushed.
     */
    public synchronized void push(LogEntry object)
    {
        while (true) {
            while (waitFlush && flushCount > 0) {
                try {
                    wait();
                } catch (InterruptedException ex) {
                }
            }
            if (queueing) {
                if (size == maxSize) {
                    if (drop) {
                        if (++tail == maxSize) {
                            tail = 0;
                        }
                        array[head++] = object;
                        if (head == maxSize) {
                            head = 0;
                        }
                        notifyAll();
                        return;
                    }
                } else {
                    array[head++] = object;
                    if (head == maxSize) {
                        head = 0;
                    }
                    size++;
                    notifyAll();
                    return;
                }
            } else if (cubby == null) {
                cubby = object;
                notifyAll();
                return;
            }
            try {
                mustPop = true;
                wait();
            } catch (InterruptedException ex) {
            }
            mustPop = false;
        }
    }

    // }}}
    // {{{ pop

    /**
     * Pop an object from the front of the queue.<P>
     */
    public synchronized LogEntry pop()
    {
        while (queueing ? (size == 0) : (cubby == null)) {
            try {
                wait();
            } catch (InterruptedException ex) {
            }
        }
        if (queueing) {
            LogEntry ret = array[tail];
            // XXX clear for gc
            array[tail++] = null;
            //  tail++;
            if (tail == maxSize) {
                tail = 0;
            }
            size--;
            //  notifyAll();
            flushCount++;
            return ret;
        } else {
            LogEntry ret = cubby;
            cubby = null;
            //  notifyAll();
            flushCount++;
            return ret;
        }
    }

    // }}}
    // {{{ popAll

    /**
     * Pop all entries from the queue.
     *
     * @return  An array of popped entries.
     */
    public synchronized LogEntry[] popAll()
    {
        while (queueing ? (size == 0) : (cubby == null)) {
            try {
                wait();
            } catch (InterruptedException ex) {
            }
        }
        LogEntry[] ret;
        if (queueing) {
            ret = new LogEntry[size];
            if (head > tail) {
                System.arraycopy(array, tail, ret, 0, size);
            } else {
                System.arraycopy(array, tail, ret, 0, maxSize - tail);
                if (head > 0) {
                    System.arraycopy(array, 0, ret, maxSize - tail, head);
                }
            }
            // XXX clear for gc
            array = new LogEntry[maxSize];
            head = tail = size = 0;
        } else {
            ret = new LogEntry[] {cubby};
            cubby = null;
        }
        //  notifyAll();
        flushCount++;
        return ret;
    }

    // }}}
    // {{{ entriesProcessed

    /**
     * Notify the LogQueue that popped entries have been processed and new
     * ones may be accepted.
     */
    public synchronized void entriesProcessed()
    {
        flushCount--;
        notifyAll();
    }

    // }}}

    //------------------------------------------------------------------------------------------
    // Changing attributes
    //------------------------------------------------------------------------------------------
    // {{{ isQueueing

    /**
     * Get the queueing state of the LogQueue.
     *
     * @return  The queueing state of the LogQueue.
     */
    public final boolean isQueueing()
    {
        return queueing;
    }

    // }}}
    // {{{ setQueueing

    /**
     * Set the queueing state of the LogQueue.
     *
     * @param   queueing        The queueing state to set.
     */
    public final synchronized void setQueueing(boolean queueing)
    {
        if (this.queueing != queueing) {
            // must be empty before changing queueing
            while (this.queueing ? size != 0 : cubby != null) {
                waitFlush = true;
                try {
                    mustPop = true;
                    wait();
                } catch (InterruptedException ex) {
                }
                mustPop = false;
            }
            this.queueing = queueing;
            notifyAll();
            while (waitFlush && flushCount > 0) {
                try {
                    wait();
                } catch (InterruptedException ex) {
                }
            }
            waitFlush = false;
        }
    }

    // }}}
    // {{{ getQueueSize

    /**
     * Get the size of the queue where it starts to either block or drop
     * entries.
     *
     * @return  The maximum size of the queue.
     */
    public final int getQueueSize()
    {
        return maxSize;
    }

    // }}}
    // {{{ setQueueSize

    /**
     * Set the maximum size of the queue.
     *
     * @param   newsize The new size of the queue.
     */
    public synchronized void setQueueSize(int newsize)
    {
        // shortcut
        if (newsize == maxSize) {
            return;
        }

        while (queueing && size > newsize) {
            if (drop) {
                tail += size - newsize;
                if (tail > maxSize) {
                    tail -= maxSize;
                }
                size = newsize;
            } else {
                try {
                    mustPop = true;
                    wait();
                } catch (InterruptedException ex) {
                }
                mustPop = false;
            }
        }

        LogEntry[] tmp = new LogEntry[newsize];
        if (queueing && size > 0) {
            if (head > tail) {
                System.arraycopy(array, tail, tmp, 0, size);
            } else {
                System.arraycopy(array, tail, tmp, 0, maxSize - tail);
                if (head > 0) {
                    System.arraycopy(array, tail, tmp, maxSize - tail, head);
                }
            }
        }
        maxSize = newsize;
        array = tmp;
        head = size;
        tail = 0;
        notifyAll();
    }

    // }}}
    // {{{ isDropOnOverflow

    /**
     * Query whether the queue drops entries when it overflows.
     *
     * @return  True if the queue drops entries on overflow.
     */
    public final boolean isDropOnOverflow()
    {
        return drop;
    }

    // }}}
    // {{{ setDropOnOverflow

    /**
     * Set whether entries should be dropped when the queue overflows. If
     * this is set to false, as is the default, the queue will block when
     * it is full, otherwise it will silently throw away old entries.
     *
     * @param   drop    Whether entries should be dropped.
     */
    public synchronized void setDropOnOverflow(boolean drop)
    {
        this.drop = drop;
        notifyAll();
    }

    // }}}

}
