// {{{ copyright

/********************************************************************
 *
 * The contents of this file are subject to the Mozilla Public
 * License Version 1.1 (the "License"); you may not use this file
 * except in compliance with the License. You may obtain a copy of
 * the License at http://www.mozilla.org/MPL/
 *
 * Software distributed under the License is distributed on an "AS
 * IS" basis, WITHOUT WARRANTY OF ANY KIND, either express or
 * implied. See the License for the specific language governing
 * rights and limitations under the License.
 *
 * The Original Code is qfs.de code.
 *
 * The Initial Developer of the Original Code is Gregor Schmid.
 * Portions created by Gregor Schmid are
 * Copyright (C) 1999 Quality First Software, Gregor Schmid.
 * All Rights Reserved.
 *
 * Contributor(s):
 *
 *******************************************************************/

// }}}

package de.qfs.lib.log;

import java.io.OutputStream;
import java.io.IOException;

/**
 * This stream logs everything written to it. It can be useful to pipe errors
 * and debugging output from third party packages to the logging
 * facilities. <p>
 *
 * A LogStream collects its input until it encouters either a LineFeed, a
 * CarriageReturn or a CarriageReturn followed by a LineFeed character.  It
 * will then log the buffered line, using the Log level, class and method
 * names passed to its constructor. An explicit call to flush will log the
 * characters collected so far. <p>
 *
 * No empty log messages will be generated.
 *
 * @author      Gregor Schmid
 */
public class LogStream extends OutputStream
{
    // {{{ variables

    /**
     * The Logger that actually handles the logging.
     */
    private Logger logger;

    /**
     * The level at which to log messages.
     */
    private int level;

    /**
     * The method name to log messages under.
     */
    private String method;

    /**
     * Internal message buffer.
     */
    private byte[] buf = new byte[1023];

    /**
     * The size of the internal buffer.
     */
    private int size = 1023;

    /**
     * The current position in the buffer.
     */
    private int pos = 0;

    /**
     * Whether the last character was a Carriage Return.
     */
    private boolean wasCR;

    /**
     * Whether the stream has been closed.
     */
    private boolean closed;

    // }}}

    // {{{ constructor

    /**
     * Create a new LogStream.
     *
     * @param   level   The log level for the logStream.
     * @param   clazz   The class name for the logged messages.
     * @param   method  The method name for the logged messages.
     *
     * @see     Log
     */
    public LogStream (int level, String clazz, String method)
    {
        this.level = level;
        this.method = method;
        logger = new Logger (clazz);
    }

    // }}}

    //----------------------------------------------------------------------
    // Public methods
    //----------------------------------------------------------------------
    // {{{ close

    /**
     * Close the LogStream.
     */
    @Override
    public synchronized void close()
    {
        try {
            flush();
        } catch (IOException ex) {
            // ignore
        }
        closed = true;
    }

    // }}}
    // {{{ flush

    /**
     * Flush the LogStream, logging the current message.
     *
     * @throws  IOException     If the LogStream has been closed.
     */
    @Override
    public synchronized void flush()
         throws IOException
    {
        if (closed) {
            throw new IOException ("closed");
        }
        log();
    }

    // }}}
    // {{{ write

    /**
     * Write one byte to the LogStream.
     *
     * @param   b       The byte to write (only the lower 8 bits of the int
     *                  are used).
     *
     * @throws  IOException     If the LogStream has been closed.
     */
    @Override
    public void write(int b)
         throws IOException
    {
        addByte((byte) b);
    }

    // }}}

    //----------------------------------------------------------------------
    // Helper methods
    //----------------------------------------------------------------------
    // {{{ addByte

    /**
     * Add a byte to the internal buffer.
     *
     * @param   b       The byte to add.
     *
     * @throws  IOException     If the stream has been closed.
     */
    private synchronized void addByte(byte b)
         throws IOException
    {
        if (closed) {
            throw new IOException ("closed");
        }
        if (b == 10 && wasCR) {
            wasCR = false;
            // ignore;
            return;
        }
        if (b == 10 || b == 13) {
            log();
            if (b == 13) {
                wasCR = true;
            }
        } else {
            if (pos == size - 1) {
                grow();
            }
            buf[pos++] = b;
        }
    }

    // }}}
    // {{{ log

    /**
     * Log a message from the buffer.
     */
    private synchronized void log()
    {
        if (pos > 0 && level >= logger.level) {
            logger.log(level, method, new String (buf, 0, pos));
        }
        pos = 0;
    }

    // }}}
    // {{{ grow

    /**
     * Grow the internal buffer.
     */
    private synchronized void grow()
    {
        byte[] tmp = new byte [2 * size];
        System.arraycopy (buf, 0, tmp, 0, pos);
        buf = tmp;
    }

    // }}}
}
