// {{{ copyright

/********************************************************************
 *
 * The contents of this file are subject to the Mozilla Public
 * License Version 1.1 (the "License"); you may not use this file
 * except in compliance with the License. You may obtain a copy of
 * the License at http://www.mozilla.org/MPL/
 *
 * Software distributed under the License is distributed on an "AS
 * IS" basis, WITHOUT WARRANTY OF ANY KIND, either express or
 * implied. See the License for the specific language governing
 * rights and limitations under the License.
 *
 * The Original Code is qfs.de code.
 *
 * The Initial Developer of the Original Code is Oliver Brandt.
 * Portions created by Oliver Brandt are
 * Copyright (C) 2000 ATecoM GmbH.
 * Portions created by Gregor Schmid are
 * Copyright (C) 2000 Quality First Software, Gregor Schmid.
 * All Rights Reserved.
 *
 * Contributor(s):
 * Gregor Schmid, Quality First Software
 *
 *******************************************************************/

// }}}

package de.qfs.lib.log;

// {{{ imports

import de.qfs.lib.log.Log;

// }}}

/**
 * A LogWriter is responsible for the final dispatch of {@link LogEntry
 * LogEntries} after they have passed through the chain of {@link LogFilter
 * LogFilters}. Possible uses are printing entries to System.err or some other
 * stream or sending them via RMI to the LogServer qflog. <p>
 *
 * A LogWriter can also be combined with a {@link LogUser LogUser} to extract
 * and distribute LogEntries directly from the filter chain. Thus filtering is
 * decoupled from distribution.
 *
 * @author      Oliver Brandt
 * @author      Gregor Schmid
 * @since       0.98.0
 * @see         Log#setLogWriter
 * @see         LevelFilter
 */
public interface LogWriter
{
    /**
     * Write one LogEntry.
     *
     * @param   entry   The entry to write.
     */
    public void write(LogEntry entry);

    /**
     * Write an array of LogEntires in one go. Clients of a LogWriter should
     * use this method in preference to {@link #write(LogEntry)
     * write(LogEntry)}, since the difference in performance can be enourmous
     * (e.g. when using RMI).
     *
     * @param   entries The entries to write.
     */
    public void write(LogEntry[] entries);

    /**
     * Close the LogWriter and free its resources, e.g. open Streams.
     */
    public void close();

    /**
     * Set the format the LogWriter should use to format LogEntries.
     *
     * @param   format  The format to use.
     */
    public void setFormat(LogFormat format);

    /**
     * Get the format the LogWriter uses to format LogEntries.
     *
     * @return  The format used by the LogWriter.
     */
    public LogFormat getFormat();

}
