// {{{ copyright

/********************************************************************
 *
 * The contents of this file are subject to the Mozilla Public
 * License Version 1.1 (the "License"); you may not use this file
 * except in compliance with the License. You may obtain a copy of
 * the License at http://www.mozilla.org/MPL/
 *
 * Software distributed under the License is distributed on an "AS
 * IS" basis, WITHOUT WARRANTY OF ANY KIND, either express or
 * implied. See the License for the specific language governing
 * rights and limitations under the License.
 *
 * The Original Code is qfs.de code.
 *
 * The Initial Developer of the Original Code is Gregor Schmid.
 * Portions created by Gregor Schmid are
 * Copyright (C) 1999 Quality First Software, Gregor Schmid.
 * All Rights Reserved.
 *
 * Contributor(s):
 *
 *******************************************************************/

// }}}

package de.qfs.lib.log;

// {{{ imports

import java.io.ByteArrayOutputStream;
import java.io.PrintWriter;

import java.lang.reflect.Array;
import java.lang.reflect.Method;

import java.util.Collection;
import java.util.ConcurrentModificationException;
import java.util.Enumeration;
import java.util.Hashtable;
import java.util.Map;
import java.util.Map.Entry;
import java.util.MissingResourceException;
import java.util.ResourceBundle;
import java.util.Set;

// }}}

// {{{ doc

/**
 * This class simplyfies logging of messages. <p>
 *
 * It encapsulates the name of its owner and uses it as the sender of the
 * messages it logs. Additionally it provides an advanced scheme to reduce
 * unwanted logging, since logging can get rather expensive in terms of memory
 * and runtime consumption if used seriously. <p>
 *
 * Typical use of a Logger looks like this:
 * <pre>
 *  // Create the Logger for SomeClass
 *  private static Logger logger = new Logger ("de.qfs.lib.log.SomeCLass");
 *
 *  public SomeType someMethod (SomeOtherType someParam)
 *  {
 *      if (logger.level >= Log.MTD) {
 *          logger.log (Log.MTD, "someMethod(SomeOtherType)",
 *                      logger.level < Log.MTDDETAIL ? "" :
 *                      "someParam: " + someParam);
 *      }
 *      ...
 * }
 * </pre>
 *
 * That way, the log message for the method entry will only be constructed, if
 * logger.level is at least Log.MTD. If it is exactly Log.MTD, no parameters
 * will be added to the log message, if it is higher, the value of someParam
 * will be logged as well. <p>
 *
 * Note that the Logger's {@link #level level} is not checked in the {@link
 * #log log} methods in order to encourage this style. It cannot be
 * overemphasized: this is the most efficient way to skip unwanted log
 * messages. Nevertheless, for those who value tidy source more than efficient
 * code, the convenience methods {@link #err err}, {@link #wrn wrn} etc. have
 * been added. These will log a message only if the Logger's level allows
 * it. <p>
 *
 * The value of logger.level can be customized through the {@link #setLogLevel
 * setLogLevel} et. al. methods. <p>
 *
 * Changes in the log levels can be monitored by adding a {@link
 * LogLevelListener LogLevelListener} via {@link #addLogLevelListener
 * addLogLevelListener}.
 *
 * @see Log
 * @author      Gregor Schmid
 */

// }}}
public class Logger
{
    // {{{ variables

    /**
     * Log level set indirectly by using {@link #setDefaultLogLevel
     * setDefaultLogLevel} etc. This attribute should be considered read-only.
     * It is public only to keep the clutter of protecting log calls to a
     * minimum.
     */
    public int level;

    /**
     * Default level for a Logger: Log.WRNDETAIL.
     */
    public final static int DEFAULT_LOG_LEVEL = Log.WRNDETAIL;

    /**
     * Educated guess for average message sized, used to initialize StringBuilders for
     * constructing messages.
     */
    private final static int AVERAGE_MESSAGE_SIZE = 200;

    /**
     * Typical length of interesting elements in an array.
     */
    private final static int DEFAULT_NUMBER_OF_ELEMENTS_TO_LOG_IN_ARRAY = 30;

    /**
     * Typical length of interesting elements in a map.
     */
    private final static int DEFAULT_NUMBER_OF_ELEMENTS_TO_LOG_IN_MAP = 50;

    /**
     * The name of the Class owning the Logger.
     */
    private String owner;

    /**
     * The characters for hexadecimal values.
     */
    private final static char[] nibbles = {
        '0','1','2','3','4','5','6','7','8','9','A','B','C','D','E','F'
    };

    /**
     * The log levels for the Loggers.
     */
    private static LogLevels levels = new LogLevels ();

    /**
     * Thread local buffer for parsing stacktrace information.
     */
    private static ThreadLocal localBOS = new ThreadLocal () {
            @Override
            protected synchronized Object initialValue() {
                return new ByteArrayOutputStream (1000);
            }
        };

    /**
     * Thread local writer for parsing stacktrace information.
     */
    private static ThreadLocal localPW = new ThreadLocal () {
            @Override
            protected synchronized Object initialValue() {
                return new PrintWriter ((ByteArrayOutputStream) localBOS.get());
            }
        };

    /**
     * Whether the stack trace shortcut available with JDK 1.4+ can be used.
     */
    private static Boolean is14;

    /**
     * Cache for Throwable.getStackTrace() method.
     */
    private static Method getStackTrace;

    /**
     * Cache for StackTraceElement.getClassName() method.
     */
    private static Method getClassName;

    /**
     * Required for reflection calls to no arg methods.
     */
    private final static Class[] noArgsArray = new Class[0];

    // }}}

    //------------------------------------------------------------------------------------------
    // Constructors
    //------------------------------------------------------------------------------------------
    // {{{ Logger(String)

    /**
     * Construct a Logger owned by owner.
     *
     * @param   owner   The name of the owner.
     */
    public Logger (String owner)
    {
        this.owner = owner;
        setLevel(getLogLevel(this));
    }

    // }}}
    // {{{ Logger(Object)

    /**
     * Construct a Logger owned by owner.
     *
     * @param   owner   The owner.
     */
    public Logger (Object owner)
    {
        this(owner.getClass().getName());
    }

    // }}}
    // {{{ Logger(Class)

    /**
     * Construct a Logger owned by owner.
     *
     * @param   owner   The class of the owner.
     */
    public Logger (Class owner)
    {
        this(owner.getName());
    }

    // }}}

    //------------------------------------------------------------------------------------------
    // Level access
    //------------------------------------------------------------------------------------------
    // {{{ setLevel

    /**
     * Set the level of the Logger. Though the level member is public for convinence it should be
     * set only using this method to enable custom Loggers to override it and react to a level
     * change.
     *
     * @param   level   The level to set.
     */
    public void setLevel(int level)
    {
        this.level = level;
    }

    // }}}

    //------------------------------------------------------------------------------------------
    // Log levels for classes
    //------------------------------------------------------------------------------------------
    // {{{ setDefaultLogLevel

    /**
     * Set the default level for all Loggers, for which no level is set
     * either directly on the class or on (parts of) its package. This not
     * only affects future calls to {@link #getLogLevel getLogLevel}, but also
     * changes the log level of already registered loggers.
     *
     * @param   level   The new default level.
     */
    public static void setDefaultLogLevel(int level)
    {
        if (levels != null) {
            levels.setDefaultLogLevel(level);
        }
    }

    // }}}
    // {{{ setLogLevel

    /**
     * Set the log level for a class or package. If the name represents a
     * package or part of a package it must end with '.' to distinguish it
     * from a class name. Otherwise setting the level for e.g.
     * "de.qfs.lib.log.Log" would implicitly set the level for
     * "de.qfs.lib.log.Logger" as well.
     *
     * @param   name    The name of the class or package (part). Package names
     *                  must end with a dot ('.').
     * @param   level   The maximum level at which objects of the class or
     *                  classes from the package should log messages.
     */
    public static void setLogLevel(String name, int level)
    {
        if (levels != null) {
            levels.setLogLevel(name, level);
        }
    }

    // }}}
    // {{{ removeLogLevel

    /**
     * Remove the level that was set for a Class or package so the
     * Class or the package's Classes will now inherit the default log level
     * or a package (part) level.
     *
     * @param   name    The name of the class or package (part). Package names
     *                  must end with a dot.
     */
    public static void removeLogLevel(String name)
    {
        if (levels != null) {
            levels.removeLogLevel(name);
        }
    }

    // }}}
    // {{{ getLogLevel

    /**
     * Get the level for a Logger. Calling this method will register the
     * Logger requesting the level, so that later changes to the default
     * level, its package's level or explicitly for the class can change the
     * Logger's log level.
     *
     * @param   logger  The Logger requesting its log level.
     *
     * @return The log level for the class owning the logger, if it has been
     *         explicitly set with {@link #setLogLevel setLogLevel}, the level
     *         of its package or the default level set with {@link
     *         #setDefaultLogLevel setDefaultLogLevel} otherwise.
     */
    public static int getLogLevel(Logger logger)
    {
        if (levels != null) {
            return levels.getLogLevel(logger);
        }
        return DEFAULT_LOG_LEVEL;
    }

    // }}}
    // {{{ setLogLevels(Hashtable)

    /**
     * Set various levels by parsing a set of properties in a Hashtable. The
     * Hashtable's keys must be of the form "log-NAME" where "NAME" is the
     * name of a Class or package as required by {@link #setLogLevel
     * setLogLevel}. The corresponding value must be a String that parses as
     * an int that will be used as the level to set. <p>
     *
     * Key/value pairs that do not follow these conventions will be silently
     * ignored. Thus for example the system properties may be used to set log
     * levels as follows:
     * <pre>
     *     Logger.setLogLevels(System.getProperties());
     * </pre>
     *
     * The default level can be set with the key "log-default".<p>
     *
     * @param   properties      The Hashtable to parse.
     */
    public static void setLogLevels(Hashtable properties)
    {
        for (Enumeration e = properties.keys(); e.hasMoreElements(); ) {
            String name = (String) e.nextElement();
            if (name.startsWith("log-")) {
                try {
                    int level = Integer.parseInt
                        ((String) properties.get(name));
                    if (name.equals("log-default")) {
                        setDefaultLogLevel(level);
                    } else {
                        setLogLevel(name.substring(4), level);
                    }
                } catch (NumberFormatException ex) {
                } catch (ClassCastException ex) {
                }
            }
        }
    }

    // }}}
    // {{{ setLogLevels(ResourceBundle)

    /**
     * Set various levels by parsing a ResourceBundle. The ResourceBundle's
     * names must be of the form "log-NAME" where "NAME" is the name of a
     * Class or package as required by {@link #setLogLevel setLogLevel}. The
     * corresponding value must parse as an int and will be used as the level
     * to set. <p>
     *
     * Resources that do not follow these conventions will be silently
     * ignored.
     *
     * The default level can be set with the resource "log-default".<p>
     *
     * @param   rb      The ResourceBundle to parse.
     */
    public static void setLogLevels(ResourceBundle rb)
    {
        for (Enumeration e = rb.getKeys(); e.hasMoreElements(); ) {
            String name = (String) e.nextElement();
            if (name.startsWith("log-")) {
                try {
                    int level = Integer.parseInt(rb.getString(name));
                    if (name.equals("log-default")) {
                        setDefaultLogLevel(level);
                    } else {
                        setLogLevel(name.substring(4), level);
                    }
                } catch (MissingResourceException ex) {
                } catch (NumberFormatException ex) {
                }
            }
        }
    }

    // }}}

    //------------------------------------------------------------------------------------------
    // LogLevelListener related methods
    //------------------------------------------------------------------------------------------
    // {{{ addLogLevelListener

    /**
     * Add a LogLevelListener for the log levels.
     *
     * @param   listener        The listener to add.
     */
    public static void addLogLevelListener(LogLevelListener listener)
    {
        levels.addLogLevelListener(listener);
    }

    // }}}
    // {{{ removeLogLevelListener

    /**
     * Remove a LogLevelListener for the log levels.
     *
     * @param   listener        The listener to remove.
     */
    public static void removeLogLevelListener(LogLevelListener listener)
    {
        levels.removeLogLevelListener(listener);
    }

    // }}}
    // {{{ getLogLevels

    /**
     * Get the current log levels. This method is needed to synchronize the
     * LogLevelListener after initialization.
     *
     * @return  An object array that contains an alternating sequence of
     *          class/package names and log levels. The levels are Integers
     *          that may be null for classes for which a level has been
     *          requested but not explicitly set.
     */
    public static Object[] getLogLevels()
    {
        return levels.getLogLevels();
    }

    // }}}
    // {{{ setLogLevel

    /**
     * Callback method for a LogLevelListener to change the log level for
     * a class or package.
     *
     * @param   source  The listener that causes the change. It will not be
     *                  notified to avoid recursion.
     * @param   name    The name of the affected class or package.
     * @param   level   The new log level.
     */
    public static void setLogLevel(LogLevelListener source,
                                   String name, int level)
    {
        levels.setLogLevel(source, name, level);
    }

    // }}}
    // {{{ removeLogLevel

    /**
     * Callback method for a LogLevelListener to remove the log level for
     * a class or package.
     *
     * @param   source  The listener that causes the change. It will not be
     *                  notified to avoid recursion.
     * @param   name    The name of the affected class or package.
     */
    public static void removeLogLevel(LogLevelListener source, String name)
    {
        levels.removeLogLevel(source, name);
    }

    // }}}

    //------------------------------------------------------------------------------------------
    // log messages and exceptions
    //------------------------------------------------------------------------------------------
    // {{{ log(int, String, String)

    /**
     * Log a message. The Logger's level is NOT checked.
     *
     * @param   level   The level of the message.
     * @param   method  The method that sent the message.
     * @param   message The message.
     */
    public void log(int level, String method, String message)
    {
        Log.log(level, System.currentTimeMillis(), owner,
                method, message);
    }

    // }}}
    // {{{ log(String, Throwable)

    /**
     * Log an Exception at leve {@link Log#ERR Log.ERR}. The Logger's level is
     * NOT checked.
     *
     * @param   method  The method that sent the message.
     * @param   throwable The Throwable.
     */
    public void log(String method, Throwable throwable)
    {
        Log.log(Log.ERR, owner, method, throwable);
    }

    // }}}
    // {{{ log(String, Throwable, String)

    /**
     * Log an Exception at level {@link Log#ERR Log.ERR}. The Logger's level
     * is NOT checked.
     *
     * @param   method  The method that sent the message.
     * @param   throwable The Throwable.
     * @param   detail  Some extra info to log with the Exception name.
     */
    public void log(String method, Throwable throwable, String detail)
    {
        Log.log(Log.ERR, owner, method, throwable, detail);
    }

    // }}}
    // {{{ log(int, String, Throwable)

    /**
     * Log an Exception at some specified level. The Logger's level is NOT
     * checked.
     *
     * @param   level   The level for the Exception.
     * @param   method  The method that sent the message.
     * @param   throwable The Throwable.
     */
    public void log(int level, String method, Throwable throwable)
    {
        Log.log(level, owner, method, throwable);
    }

    // }}}
    // {{{ log(int, String, Throwable, String)

    /**
     * Log an Exception at some specified level. The Logger's level is NOT
     * checked.
     *
     * @param   level   The level for the Exception.
     * @param   method  The method that sent the message.
     * @param   throwable The Throwable.
     * @param   detail  Some extra info to log with the Exception name.
     */
    public void log(int level, String method, Throwable throwable,
                          String detail)
    {
        Log.log(level, owner, method, throwable, detail);
    }

    // }}}
    // {{{ dumpStack

    /**
     * Log a stack trace. The Logger's level is NOT checked.
     *
     * @param   level   The level to log the stack trace at.
     * @param   method  The method that sent the message.
     * @param   msg     The message to log with the stack trace.
     */
    public void dumpStack(int level, String method, String msg)
    {
        try {
            throw new Log.DumpStackException (msg);
        } catch (final Exception ex) {
            log(level, method, ex);
        }
    }

    // }}}
    //------------------------------------------------------------------------------------------
    // Alternative logging based on a message builder
    //------------------------------------------------------------------------------------------
    // {{{ build

    /**
     * Create a new log message builder. The loggers level must be checked before starting to
     * build.
     *
     * @deprecated  Use {@link QFLogger#lvlBuild(int, String)} instead.
     *
     * @param   method  The method to log for.
     *
     * @return  The new builder.
     */
    @Deprecated
    public Builder build(String method)
    {
        return new Builder (method);
    }

    // }}}
    // {{{ class Builder

    /**
     * Helper class for building log messages.
     */
    public class Builder
    {
        // {{{ variables

        /**
         * The buffer used for building the string.
         */
        private StringBuilder sb;

        /**
         * The method for which the message is built.
         */
        private String method;

        // }}}

        //--------------------------------------------------------------------------------------
        // Constructor
        //--------------------------------------------------------------------------------------
        // {{{ Builder

        /**
         * Create a new Builder.
         *
         * @param   method  The method to log for.
         */
        public Builder (String method)
        {
            this.method = method;
            sb = new StringBuilder(AVERAGE_MESSAGE_SIZE);
        }

        // }}}

        // {{{ add(boolean)

        /**
         * Add a boolean to the log message.
         *
         * @param       val       The boolean to add.
         *
         * @return      The builder itself for chaining.
         */
        public final Builder add(boolean val)
        {
            sb.append(val);
            return this;
        }

        // }}}
        // {{{ add(byte)

        /**
         * Add a byte to the log message.
         *
         * @param       val       The byte to add.
         *
         * @return      The builder itself for chaining.
         */
        public final Builder add(byte val)
        {
            sb.append(val);
            return this;
        }

        // }}}
        // {{{ add(char)

        /**
         * Add a char to the log message.
         *
         * @param       val       The char to add.
         *
         * @return      The builder itself for chaining.
         */
        public final Builder add(char val)
        {
            sb.append(val);
            return this;
        }

        // }}}
        // {{{ add(int)

        /**
         * Add an int to the log message.
         *
         * @param       val       The int to add.
         *
         * @return      The builder itself for chaining.
         */
        public final Builder add(int val)
        {
            sb.append(val);
            return this;
        }

        // }}}
        // {{{ addHex(int)

        /**
         * Add an int in hexadecimal notation to the log message.
         *
         * @param       val       The int to add.
         *
         * @return      The builder itself for chaining.
         */
        public final Builder addHex(int val)
        {
            sb.append("0x");
            sb.append(Integer.toHexString(val));
            return this;
        }

        // }}}
        // {{{ addHex(Integer)

        /**
         * Add an Integer in hexadecimal notation to the log message.
         *
         * @param       val       The int to add.
         *
         * @return      The builder itself for chaining.
         */
        public final Builder addHex(Integer val)
        {
            if (val != null) {
                return addHex(val.intValue());
            }
            sb.append("null");
            return this;
        }

        // }}}
        // {{{ add(long)

        /**
         * Add a long to the log message.
         *
         * @param       val       The long to add.
         *
         * @return      The builder itself for chaining.
         */
        public final Builder add(long val)
        {
            sb.append(val);
            return this;
        }

        // }}}
        // {{{ addHex(long)

        /**
         * Add an long in hexadecimal notation to the log message.
         *
         * @param       val       The long to add.
         *
         * @return      The builder itself for chaining.
         */
        public final Builder addHex(long val)
        {
            sb.append("0x");
            sb.append(Long.toHexString(val));
            return this;
        }

        // }}}
        // {{{ addHex(Long)

        /**
         * Add a Long in hexadecimal notation to the log message.
         *
         * @param       val       The long to add.
         *
         * @return      The builder itself for chaining.
         */
        public final Builder addHex(Long val)
        {
            if (val != null) {
                return addHex(val.longValue());
            }
            sb.append("null");
            return this;
        }

        // }}}
        // {{{ add(double)

        /**
         * Add a double to the log message.
         *
         * @param       val       The long to add.
         *
         * @return      The builder itself for chaining.
         */
        public final Builder add(double val)
        {
            sb.append(val);
            return this;
        }

        // }}}
        // {{{ add(Object)

        /**
         * Add an Object to the log message. If the object is an array or iterable, the first 30 elements
         * are added to the log.
         *
         * @param       val       The Object to add.
         *
         * @return      The builder itself for chaining.
         */
        public Builder add(final Object val)
        {
            if (val == null) {
                sb.append(val);
            } else {
                try {
                    if (val.getClass().isArray()) {
                        return addArray(val);
                    } else if (val instanceof Iterable<?>) {
                        return addIterable((Iterable<?>) val);
                    } else if (val instanceof Map<?,?>) {
                        return addMap((Map<?,?>) val);
                    }

                    sb.append(val);
                } catch (Throwable ex) {
                    Log.log(Log.WRN, owner, method, ex);
                    sb.append("EXCEPTED");
                }
            }
            return this;
        }

        // }}}
        // {{{ add(String)

        /**
         * Add a String to the log message.
         *
         * @param       val       The String to add.
         *
         * @return      The builder itself for chaining.
         */
        public final Builder add(String val)
        {
            sb.append(val);
            return this;
        }

        // }}}
        // {{{ addArray(Object)

        /**
         * Add the first 15 objects in an array of Objects to the log message.
         * Use {@link #setNumberOfArrayElementsToLog(int)} to change the
         * number of logged elements.
         *
         * @param       array       The objects to add.
         *
         * @return      The builder itself for chaining.
         */
        private Builder addArray(final Object array)
        {
            assert array != null;
            assert array.getClass().isArray();

            int arrayLength = Array.getLength(array);
            int hiddenElements =
                    Math.max(0,arrayLength - DEFAULT_NUMBER_OF_ELEMENTS_TO_LOG_IN_ARRAY);

            sb.append("[");
            for (int i = 0; i < (arrayLength - hiddenElements); i++) {
                if (i > 0) {
                    sb.append(',');
                }
                final Object element = Array.get(array, i);
                add(element == array ? "(this Array)" : element);
            }
            if (hiddenElements > 0) {
                sb.append(",... (")
                  .append(hiddenElements)
                  .append(" more)");
            }
            sb.append("]");
            return this;
        }

        // }}}
        // {{{ addIterable(Iterable)

        /**
         * Add the first 15 objects in a Iterable of Objects to the log message.
         * Use {@link #setNumberOfArrayElementsToLog(int)} to change the
         * number of logged elements.
         *
         * @param       val       The objects to add.
         *
         * @return      The builder itself for chaining.
         */
        @SuppressWarnings("rawtypes")
        private Builder addIterable(final Iterable val)
        {
            if (val == null) {
                sb.append(val);
            } else {

                sb.append("[");
                int i = 0;
                try {
                    for (Object element : val) {
                        if (i > 0) {
                            sb.append(',');
                        }
                        ++i;
                        if (i > DEFAULT_NUMBER_OF_ELEMENTS_TO_LOG_IN_ARRAY) {
                            sb.append("...");
                            if (val instanceof Collection) {
                                int listLength = ((Collection) val).size();
                                int hiddenElements =
                                        listLength - DEFAULT_NUMBER_OF_ELEMENTS_TO_LOG_IN_ARRAY;
                                sb.append(String.format(" (%d more)", hiddenElements));
                            }
                            break;
                        }
                        add(element == val ? "(this Iterable)" : element);
                    }
                } catch (final ConcurrentModificationException ex) {
                    sb.append(" ConcurrentModification!");
                }
                sb.append("]");
            }
            return this;
        }
        // }}}
        // {{{ addMap(Map)

        /**
         * Add the first 15 objects in a Map of Objects to the log message.
         * Use {@link #setNumberOfArrayElementsToLog(int)} to change the
         * number of logged elements.
         *
         * @param       val       The objects to add.
         *
         * @return      The builder itself for chaining.
         */
        @SuppressWarnings("rawtypes")
        private Builder addMap(final Map val)
        {
            if (val == null) {
                sb.append(val);
            } else {

                sb.append("{");
                try {
                    int i = 0;
                    final Set<Entry<Object,Object>> entrySet = val.entrySet();
                    for(final Entry<Object,Object> entry: entrySet) {
                        if (i > 0) {
                            sb.append(',');
                        }
                        ++i;
                        if (i > DEFAULT_NUMBER_OF_ELEMENTS_TO_LOG_IN_MAP) {
                            sb.append("...");
                            int listLength = entrySet.size();
                            int hiddenElements =
                                    listLength - DEFAULT_NUMBER_OF_ELEMENTS_TO_LOG_IN_MAP;
                            sb.append(String.format(" (%d more)",hiddenElements));
                            break;
                        }

                        final Object key = entry.getKey();
                        final Object value = entry.getValue();

                        add(key == val ? "(this Map)" : key);
                        sb.append("=");
                        add(value == val ? "(this Map)" : value);
                    }
                } catch (final ConcurrentModificationException ex) {
                    sb.append(" ConcurrentModification!");
                }
                sb.append("}");
            }
            return this;
        }
        // }}}
        // {{{ addAll(Object)

        /**
         * If the object is an array, add all of its elements, otherwise add the object itself.
         *
         * @param       val       The object(s) to add.
         *
         * @return      The builder itself for chaining.
         */
        public final Builder addAll(Object val)
        {
            if (val instanceof Object[]) {
                return addAll((Object[]) val);
            }
            if (val instanceof int[]) {
                return addAll((int[]) val);
            }
            if (val instanceof long[]) {
                return addAll((long[]) val);
            }
            sb.append(val);
            return this;
        }

        // }}}
        // {{{ addAll(Object[])

        /**
         * Add each object in an array of Objects to the log message.
         *
         * @param       val       The objects to add.
         *
         * @return      The builder itself for chaining.
         */
        public final Builder addAll(Object[] val)
        {
            if (val == null) {
                sb.append((String) null);
            } else {
                sb.append("[");
                for (int i = 0; i < val.length; i++) {
                    if (i > 0) {
                        sb.append(',');
                    }
                    addAll(val[i]);
                }
                sb.append("]");
            }
            return this;
        }

        // }}}
        // {{{ addAll(int[])

        /**
         * Add each int in an array of ints to the log message.
         *
         * @param       val       The ints to add.
         *
         * @return      The builder itself for chaining.
         */
        public final Builder addAll(int[] val)
        {
            if (val == null) {
                sb.append((String) null);
            } else {
                sb.append("[");
                for (int i = 0; i < val.length; i++) {
                    if (i > 0) {
                        sb.append(',');
                    }
                    sb.append(val[i]);
                }
                sb.append("]");
            }
            return this;
        }

        // }}}
        // {{{ addAll(long[])

        /**
         * Add each long in an array of longs to the log message.
         *
         * @param       val       The longs to add.
         *
         * @return      The builder itself for chaining.
         */
        public final Builder addAll(long[] val)
        {
            if (val == null) {
                sb.append((String) null);
            } else {
                sb.append("[");
                for (int i = 0; i < val.length; i++) {
                    if (i > 0) {
                        sb.append(',');
                    }
                    sb.append(val[i]);
                }
                sb.append("]");
            }
            return this;
        }

        // }}}
        // {{{ log

        /**
         * Log the message.
         *
         * @param       level   The level to log at.
         */
        public void log(int level)
        {
            Logger.this.log(level, method, sb.toString());
        }

        // }}}
        // {{{ dumpStack

        /**
         * Log the message including a stack dump.
         *
         * @param       level   The level to log at.
         */
        public void dumpStack(int level)
        {
            Logger.this.dumpStack(level, method, sb.toString());
        }

        // }}}
    }

    // }}}

    //------------------------------------------------------------------------------------------
    // Convenience methods
    //------------------------------------------------------------------------------------------
    // {{{ err(String, String)

    /**
     * Log a message at level {@link Log#ERR Log.ERR}, if the Logger's level
     * allows it.
     *
     * @param   method  The method that sent the message.
     * @param   message The message.
     *
     * @since   0.98.1
     */
    public final void err(String method, String message)
    {
        if (level >= Log.ERR) {
            Log.log(Log.ERR, System.currentTimeMillis(), owner,
                    method, message);
        }
    }

    // }}}
    // {{{ err(String, Throwable)

    /**
     * Log an Exception at level {@link Log#ERR Log.ERR}, if the Logger's level
     * allows it.
     *
     * @param   method  The method that sent the message.
     * @param   throwable The Throwable.
     *
     * @since   0.98.1
     */
    public final void err(String method, Throwable throwable)
    {
        if (level >= Log.ERR) {
            Log.log(Log.ERR, owner, method, throwable);
        }
    }

    // }}}
    // {{{ err(String, Throwable, String)

    /**
     * Log an Exception.
     *
     * @param   method  The method that sent the message.
     * @param   throwable The Throwable.
     * @param   detail  Some extra info to log with the Exception name.
     *
     * @since   0.98.1
     */
    public final void err(String method, Throwable throwable, String detail)
    {
        if (level >= Log.ERR) {
            Log.log(Log.ERR, owner, method, throwable, detail);
        }
    }

    // }}}
    // {{{ wrn(String, String)

    /**
     * Log a message at level {@link Log#WRN Log.WRN}, if the Logger's level
     * allows it.
     *
     * @param   method  The method that sent the message.
     * @param   message The message.
     *
     * @since   0.98.1
     */
    public final void wrn(String method, String message)
    {
        if (level >= Log.WRN) {
            Log.log(Log.WRN, System.currentTimeMillis(), owner,
                    method, message);
        }
    }

    // }}}
    // {{{ wrn(String, Throwable)

    /**
     * Log an Exception at level {@link Log#WRN Log.WRN}, if the Logger's level
     * allows it.
     *
     * @param   method  The method that sent the message.
     * @param   throwable The Throwable.
     *
     * @since   0.98.1
     */
    public final void wrn(String method, Throwable throwable)
    {
        if (level >= Log.WRN) {
            Log.log(Log.WRN, owner, method, throwable);
        }
    }

    // }}}
    // {{{ wrn(String, Throwable, String)

    /**
     * Log an Exception.
     *
     * @param   method  The method that sent the message.
     * @param   throwable The Throwable.
     * @param   detail  Some extra info to log with the Exception name.
     *
     * @since   0.98.1
     */
    public final void wrn(String method, Throwable throwable, String detail)
    {
        if (level >= Log.WRN) {
            Log.log(Log.WRN, owner, method, throwable, detail);
        }
    }

    // }}}
    // {{{ msg(String, String)

    /**
     * Log a message at level {@link Log#MSG Log.MSG}, if the Logger's level
     * allows it.
     *
     * @param   method  The method that sent the message.
     * @param   message The message.
     *
     * @since   0.98.1
     */
    public final void msg(String method, String message)
    {
        if (level >= Log.MSG) {
            Log.log(Log.MSG, System.currentTimeMillis(), owner,
                    method, message);
        }
    }

    // }}}
    // {{{ msg(String, Throwable)

    /**
     * Log an Exception at level {@link Log#MSG Log.MSG}, if the Logger's level
     * allows it.
     *
     * @param   method  The method that sent the message.
     * @param   throwable The Throwable.
     *
     * @since   0.98.1
     */
    public final void msg(String method, Throwable throwable)
    {
        if (level >= Log.MSG) {
            Log.log(Log.MSG, owner, method, throwable);
        }
    }

    // }}}
    // {{{ msg(String, Throwable, String)

    /**
     * Log an Exception.
     *
     * @param   method  The method that sent the message.
     * @param   throwable The Throwable.
     * @param   detail  Some extra info to log with the Exception name.
     *
     * @since   0.98.1
     */
    public final void msg(String method, Throwable throwable, String detail)
    {
        if (level >= Log.MSG) {
            Log.log(Log.MSG, owner, method, throwable, detail);
        }
    }

    // }}}
    // {{{ mtd(String, String)

    /**
     * Log a message at level {@link Log#MTD Log.MTD}, if the Logger's level
     * allows it.
     *
     * @param   method  The method that sent the message.
     * @param   message The message.
     *
     * @since   0.98.1
     */
    public final void mtd(String method, String message)
    {
        if (level >= Log.MTD) {
            Log.log(Log.MTD, System.currentTimeMillis(), owner,
                    method, message);
        }
    }

    // }}}
    // {{{ mtd(String, Throwable)

    /**
     * Log an Exception at level {@link Log#MTD Log.MTD}, if the Logger's level
     * allows it.
     *
     * @param   method  The method that sent the message.
     * @param   throwable The Throwable.
     *
     * @since   0.98.1
     */
    public final void mtd(String method, Throwable throwable)
    {
        if (level >= Log.MTD) {
            Log.log(Log.MTD, owner, method, throwable);
        }
    }

    // }}}
    // {{{ mtd(String, Throwable, String)

    /**
     * Log an Exception.
     *
     * @param   method  The method that sent the message.
     * @param   throwable The Throwable.
     * @param   detail  Some extra info to log with the Exception name.
     *
     * @since   0.98.1
     */
    public final void mtd(String method, Throwable throwable, String detail)
    {
        if (level >= Log.MTD) {
            Log.log(Log.MTD, owner, method, throwable, detail);
        }
    }

    // }}}
    // {{{ dbg(String, String)

    /**
     * Log a message at level {@link Log#DBG Log.DBG}, if the Logger's level
     * allows it.
     *
     * @param   method  The method that sent the message.
     * @param   message The message.
     *
     * @since   0.98.1
     */
    public final void dbg(String method, String message)
    {
        if (level >= Log.DBG) {
            Log.log(Log.DBG, System.currentTimeMillis(), owner,
                    method, message);
        }
    }

    // }}}
    // {{{ dbg(String, Throwable)

    /**
     * Log an Exception at level {@link Log#DBG Log.DBG}, if the Logger's level
     * allows it.
     *
     * @param   method  The method that sent the message.
     * @param   throwable The Throwable.
     *
     * @since   0.98.1
     */
    public final void dbg(String method, Throwable throwable)
    {
        if (level >= Log.DBG) {
            Log.log(Log.DBG, owner, method, throwable);
        }
    }

    // }}}
    // {{{ dbg(String, Throwable, String)

    /**
     * Log an Exception.
     *
     * @param   method  The method that sent the message.
     * @param   throwable The Throwable.
     * @param   detail  Some extra info to log with the Exception name.
     *
     * @since   0.98.1
     */
    public final void dbg(String method, Throwable throwable, String detail)
    {
        if (level >= Log.DBG) {
            Log.log(Log.DBG, owner, method, throwable, detail);
        }
    }

    // }}}

    //------------------------------------------------------------------------------------------
    // Others
    //------------------------------------------------------------------------------------------
    // {{{ getOwnerName

    /**
     * Get the name of the Class owning this Logger.
     *
     * @return  The owner's name.
     */
    public String getOwnerName()
    {
        return owner;
    }

    // }}}
}
