// {{{ copyright

/********************************************************************
 *
 * The contents of this file are subject to the Mozilla Public
 * License Version 1.1 (the "License"); you may not use this file
 * except in compliance with the License. You may obtain a copy of
 * the License at http://www.mozilla.org/MPL/
 *
 * Software distributed under the License is distributed on an "AS
 * IS" basis, WITHOUT WARRANTY OF ANY KIND, either express or
 * implied. See the License for the specific language governing
 * rights and limitations under the License.
 *
 * The Original Code is qfs.de code.
 *
 * The Initial Developer of the Original Code is Gregor Schmid.
 * Portions created by Gregor Schmid are
 * Copyright (C) 1999 Quality First Software, Gregor Schmid.
 * All Rights Reserved.
 *
 * Contributor(s):
 *
 *******************************************************************/

// }}}

package de.qfs.lib.log;

// {{{ imports

import java.io.ByteArrayOutputStream;
import java.io.File;
import java.io.IOException;
import java.io.InputStream;
import java.io.OutputStream;
import java.io.PrintStream;

import java.util.Scanner;
import java.util.regex.Matcher;
import java.util.regex.Pattern;

import lombok.EqualsAndHashCode;
import lombok.Getter;
import lombok.NonNull;
import lombok.RequiredArgsConstructor;
import lombok.Setter;
import lombok.extern.qfs.QFLog;

import de.qfs.lib.option.TextOption;


// }}}

/**
 * Helper object to transform crypted stacktraces back ("retrace").
 *
 * <p>Requires {@code proguard.jar} and {@code retrace.jar} at some known or specified location.
 * <li>.
 * <li>lib/
 * <li>../lib/
 * <li>~/nprj/lib
 *
 * <p>Download: https://www.qftest.com/pub/proguard.zip
 *
 * @author Pascal Bihler
 *
 */
@QFLog
@RequiredArgsConstructor
@EqualsAndHashCode
public class ProguardMap {
    // {{{ constants

    private static final File[] DEV_LIB_DIR = new File[] {
            new File("."),
            new File("lib"),
            new File("../lib"),
            new File(System.getenv("QFS"),"nprj/lib"),
            new File(System.getProperty("user.home"),"nprj/lib"),
            new File("H:\\nprj\\lib"),
            new File("C:\\qfs\\nprj\\lib")
    };

    private static final File[] DEV_MAP_DIR = new File[] {
            new File(System.getenv("QFS"),"nprj/release/maps"),
            new File(System.getProperty("user.home"),"nprj/release/maps"),
            new File("H:\\nprj\\release\\maps"),
            new File("C:\\qfs\\nprj\\release\\maps")
    };

    final static public String[] CLASSPATH = new String[]{"proguard.jar","retrace.jar"};
    final static protected String RETRACE_CLASS = "proguard.retrace.ReTrace";

    final static protected Pattern TRACE_LINE_PATTERN = Pattern.compile(".*?\\bat\\s+");

    volatile Boolean libsExists = null;

    // }}}
    // {{{ variables

    @Getter final File libDir;
    @NonNull @Getter @Setter File mapFile;
    // }}}
    // {{{ main constructor

    /**
     * Tries to autodetect the folder of proguard jars
     * @param mapFile
     */
    public ProguardMap(final File mapFile) {
        this.mapFile = mapFile;

        final File detectLibDir = detectLibDir();
        this.libDir = detectLibDir == null ? new File(".") : detectLibDir;

    }

    // }}}

    // {{{ detectLibDir

    /**
     * Tries to detect the proguard library folder
     * @return
     */
    public static File detectLibDir() {
        for (final File folder: DEV_LIB_DIR) {
            qflog(DBG,"Checking folder: ", folder);
            if (checkLibs(folder)) {
                qflog(DBG,"Folder found: ", folder);
                return folder;
            }
        }
        return null;
    }


    // }}}
    // {{{ checkLibs

    /**
     * Checks, whether the required jars can be found at the {@link #libDir}
     */
    public boolean checkLibs() {
        if (libsExists == null) {
            libsExists = checkLibs(libDir);
        }
        return libsExists;
    }


    // }}}
    // {{{ checkLibs(File)

    protected static boolean checkLibs(final File libDir) {
        for (final String jarFile: CLASSPATH) {
            if (! new File(libDir,jarFile).exists()) {
                return false;
            }
        }
        return true;
    }

    // }}}
    // {{{ isStackTrace(String)

    /**
     * Checks, if a given String contains a stacktrace
     *
     * @param candidate
     * @return
     */
    public static boolean isStackTrace(@NonNull final String candidate) {
        final Matcher m = TRACE_LINE_PATTERN.matcher(candidate);
        return m.find();
    }

    // }}}
    // {{{ retrace(String)

    /**
     * Retraces the given stacktrace with the specified map file
     * @param trace
     * @return
     */
    @NonNull
    public String retrace(@NonNull final String trace) {
        if (! checkLibs()) {
            return trace;
        }

        Process processRun = null;
        try {
            processRun = Runtime.getRuntime().exec(new String[]{ "java",
                    "-cp",
                    TextOption.join(CLASSPATH, File.pathSeparator),
                    RETRACE_CLASS,
                    mapFile.getAbsolutePath()},
                    null,libDir);

        } catch (final IOException e) {
            qflog(ERR,e);
        }
        if (processRun == null) {
            return trace;
        }

        try {
            final ByteArrayOutputStream bos = new ByteArrayOutputStream();
            PrintStream in = new PrintStream(bos);

            inheritIO(processRun.getErrorStream(), System.err);
            inheritIO(processRun.getInputStream(), in);

            final OutputStream out = processRun.getOutputStream();
            // After https://github.com/facebook/proguard/blob/master/src/proguard/retrace/ReTrace.java
            // retrace expects UTF-8 input
            out.write(trace.getBytes("UTF-8"));
            out.close();

            processRun.waitFor();
            // retrace is outputting everything by using UTF-8
            return bos.toString("UTF-8");
        } catch (final Exception e) {
            qflog(ERR,e);
            return trace;
        }
    }

    // }}}
    // {{{ inheritIO(InputStream,PrintStream)

    private static void inheritIO(final InputStream src, final PrintStream dest) {
        new Thread(new Runnable() {
            @Override
            public void run() {
                Scanner sc = null;
                try {
                    sc = new Scanner(src);
                    while (sc.hasNextLine()) {
                        dest.println(sc.nextLine());
                    }
                } finally {
                    if (sc != null) {
                        sc.close();
                    }
                }
            }
        }).start();
    }

    // }}}
    // {{{ inputStreamToString(InputStream)

    @NonNull
    private static String inputStreamToString(@NonNull final InputStream is) throws Exception {
        Scanner sc = null;
        try {
            sc = new Scanner(is);
            sc.useDelimiter("\\A");
            return sc.hasNext() ? sc.next() : "";
        } finally {
            if (sc != null) {
                sc.close();
            }
        }
    }


    // }}}
    // {{{ getMapFileChooserStartDir(ProguardMap)

    @NonNull
    public static File getMapFileChooserStartDir(final ProguardMap currentMap) {
        File startDir = new File (new File (".").getAbsolutePath());
        if (currentMap != null) {
            startDir = currentMap.getMapFile().getParentFile();
        } else {
            for (final File mapDir: DEV_MAP_DIR) {
                if (mapDir.exists()) {
                    startDir = mapDir;
                    break;
                }
            }
        }
        return startDir;
    }

    // }}}

}
