// {{{ copyright

/********************************************************************
 *
 * The contents of this file are subject to the Mozilla Public
 * License Version 1.1 (the "License"); you may not use this file
 * except in compliance with the License. You may obtain a copy of
 * the License at http://www.mozilla.org/MPL/
 *
 * Software distributed under the License is distributed on an "AS
 * IS" basis, WITHOUT WARRANTY OF ANY KIND, either express or
 * implied. See the License for the specific language governing
 * rights and limitations under the License.
 *
 * The Original Code is qfs.de code.
 *
 * The Initial Developer of the Original Code is Gregor Schmid.
 * Portions created by Gregor Schmid are
 * Copyright (C) 1999 Quality First Software, Gregor Schmid.
 * All Rights Reserved.
 *
 * Contributor(s):
 *
 *******************************************************************/

// }}}


package de.qfs.lib.log;


import java.util.LinkedList;
import java.util.List;

/**
 * This class extends {@link Logger} with {@link #lvlBuild(int, String)}
 *
 * @author qfs, pb
 *
 */
// {{{ class QFLogger
public class QFLogger
    extends Logger implements QFLoggerInterface
{
    // {{{ static initializer

    static {
        // Workaround for class loading issues in case an application including qflog is started outside
        // of QF-test first and the agent tries to attach later, causing part of the log classe to be then
        // picked up by the bootstrap class loader
        final ClassLoader loader = QFLogger.class.getClassLoader();
        if (loader != null) {
            for (String cl: new String[] {
                    "DefaultLogFormat",
                    "FileLogWriter",
                    "FlushBuffer",
                    "GenericLevelAwareLogBuilder",
                    "LevelAwareLogBuilder",
                    "LevelFilter",
                    "Log",
                    "Log$DumpStackException",
                    "LogEntry",
                    "LogFilter",
                    "LogFormat",
                    "Logger$Builder",
                    "Logger",
                    "LogLevelCallback",
                    "LogLevelEvent",
                    "LogLevelListener",
                    "LogLevels",
                    "LogLevels$LogLevelThread",
                    "LogLevels$Node",
                    "Log$LogThread",
                    "LogQueue",
                    "LogSource",
                    "LogStream",
                    "LogUser",
                    "LogWriter",
                    "ProguardMap",
                    "ProguardMapPossessor",
                    "QFLogger$BridgingFilter",
                    "QFLogger$BridgingLogger",
                    "QFLogger",
                    "QFLoggerInterface",
                    "RingFileLogWriter",
                    "StreamFilter",
                    "StreamLogWriter",
                    "TreeFilter"
                }) {
                try {
                    loader.loadClass("de.qfs.lib.log." + cl);
                    for (int i = 1; i < 10; i++) {
                        try {
                            loader.loadClass("de.qfs.lib.log." + cl + "$" + i);
                        } catch (Throwable ex) {
                            break;
                        }
                    }
                } catch (Throwable ex) {
                    ex.printStackTrace();
                    break;
                }
            }
        }
    };

    // }}}
    // {{{ variables

    /**
     * Set only after the superclass constructor is done.
     */
    private boolean initialized = true;

    protected volatile List<LogObjectDumper> objectDumpers;

    // }}}

    //---------------------------------------------------------------------------------------
    // Constructors
    //---------------------------------------------------------------------------------------
    // {{{ QFLogger(Class<?>)

    public QFLogger(final Class<?> owner) {
        this (owner.getName());
    }

    // }}}
    // {{{ QFLogger(Object)

    public QFLogger(final Object owner) {
        this (owner.getClass().getName());
    }

    // }}}
    // {{{ QFLogger(String)

    public QFLogger(final String owner) {
        super (owner);
    }

    // }}}

    //---------------------------------------------------------------------------------------
    // Alternative logging based on a level aware message builder for cleaner code
    //---------------------------------------------------------------------------------------
    // {{{ lvlBuild

    /**
     * Create a new log message builder. The loggers level must be checked before starting to
     * build.
     *
     * @param   level   The level for which log statements should be created
     * @param   method  The method to log for.
     *
     * @return  The new builder
     */
    public LevelAwareLogBuilder lvlBuild(final int level, final String method)
    {
        return lvlBuild(level, method, 0);
    }

    // }}}
    // {{{ lvlBuild

    /**
     * Create a new log message builder. The loggers level must be checked before starting to
     * build.
     *
     * @param   level   The level for which log statements should be created
     * @param   method  The method to log for.
     * @param   lineNumber  The lineNumber to reference.
     *
     * @return  The new builder
     */
    public LevelAwareLogBuilder lvlBuild(final int level, final String method, final int lineNumber)
    {
        try {
            return new LevelAwareLogBuilder (this, level, method, lineNumber, objectDumpers);
        } catch (Error ex) {
            //System.err.println("builder source: " + GenericLevelAwareLogBuilder.class.getCodeSource());
            throw ex;
        }
    }

    // }}}
    // {{{ addObjectDumper

    // Only indirectly called from ObjectDumper.register
    void addObjectDumper(final LogObjectDumper objectDumper) {
        if (objectDumper == null) return;

        if (objectDumpers == null) {
            objectDumpers = new LinkedList<>();
        }
        objectDumpers.add(objectDumper);
    }

    // }}}

    //---------------------------------------------------------------------------------------
    // Level checks for methods
    //---------------------------------------------------------------------------------------
    // {{{ setLevel

    /**
     * Set the level of the Logger. Though the level member is public for conveninence it should
     * be set only using this method to enable custom Loggers to override it and react to a level
     * change.
     *
     * @param   level   The level to set.
     */
    @Override
    public void setLevel(int level)
    {
        super.setLevel(level);
        if (! initialized) {
            // Called from super class constructor.
            return;
        }
        final String mpkg = getOwnerName() + "_mtd.";
        // Check if there's a log level for the method package matching this logger's name in
        // which case we activate method-based log levels. Unfortunately it's a bit tricky to
        // get at the LogLevels here...
        boolean gotMpkg = false;
        final Object[] levels = getLogLevels();
        for (int i = 0; i < levels.length; i += 2) {
            if (mpkg.equals(levels[i])) {
                // Got package level - activate method log
                gotMpkg = true;
                break;
            }
        }
    }

    // }}}
    // {{{ setMethodLevel

    /**
     * NOOP, left for mixed qflib-usage situations only
     *
     * @return  this for chaining.
     */
    @Deprecated
    public QFLogger setMethodLevel(String method, final int level)
    {
        return this;
    }

    // }}}
    // {{{ bridgeJavaLogger(Class)

    /**
     * Attaches a filter to a javaLogger in order to pass the messages to a corresponding QF logger
     *
     * @param classToLog the class of which the corresponding logger should be bridged
     */
    public static void bridgeJavaLogger(final String classToLog)
    {
        bridgeJavaLogger(classToLog, false, 0);
    }

    // }}}
    // {{{ bridgeJavaLogger(Class, boolean)

    /**
     * Attaches a filter to a javaLogger in order to pass the messages to a corresponding QF logger
     *
     * @param classToLog the class of which the corresponding logger should be bridged
     * @param passThrough set to true to pass the messages to the java logger through, false otherwise
     */
    public static void bridgeJavaLogger(final String classToLog, final boolean passThrough)
    {
        bridgeJavaLogger(classToLog, passThrough, 0);
    }

    // }}}
    // {{{ bridgeJavaLogger(Class, boolean, int)

    /**
     * Attaches a filter to a javaLogger in order to pass the messages to a corresponding QF logger
     *
     * @param classToLog the class of which the corresponding logger should be bridged
     * @param passThrough set to true to pass the messages to the java logger through, false otherwise
     */
    public static void bridgeJavaLogger(final String classToLog, final boolean passThrough, final int levelOffset)
    {
        try {
            new BridgingFilter(classToLog, passThrough, levelOffset).bridge();
        } catch (final Throwable ex) {
            Log.log(Log.MSG, classToLog, "<setupLogger>", ex);
        }
    }

    // }}}
    // {{{ BridgingLogger

    private static class BridgingLogger extends QFLogger {

        final java.util.logging.Logger javalogger;

        public BridgingLogger(final java.util.logging.Logger javalogger, final String owner)
        {
            super(owner);
            this.javalogger = javalogger;
        }


        // {{{ setLevel()

        @Override
        public void setLevel(final int level)
        {
            super.setLevel(level);
            setJavaLoggerLevel();
        }

        // }}}
        // {{{ setJavaLoggerLevel

        public void setJavaLoggerLevel()
        {
            if (this.javalogger != null) {
                this.javalogger.setLevel(qfLoggerLevelToJavaLogLevel(this.level));
            }
        }

        // }}}
    }
    // }}}
    // {{{ BridgingFilter

    private static class BridgingFilter
        implements java.util.logging.Filter
    {
        final String classToLog;
        final boolean passThrough;
        final java.util.logging.Logger javalogger;
        final BridgingLogger qfLogger;
        final int levelOffset;

        public BridgingFilter(final String classToLog, final boolean passThrough, final int levelOffset)
        {
            super();
            this.classToLog = classToLog;
            this.passThrough = passThrough;
            this.levelOffset = levelOffset;

            this.javalogger = java.util.logging.Logger.getLogger(classToLog);
            this.qfLogger = new BridgingLogger(this.javalogger,classToLog);
            this.qfLogger.setJavaLoggerLevel();
        }

        public BridgingFilter bridge() {
            this.javalogger.setFilter(this);
            return this;
        }

        @Override
        public boolean isLoggable(java.util.logging.LogRecord record)
        {
            int qfLoggerLevel = javaLoggerLevelToQFLogLevel(record.getLevel()) + levelOffset;
            if (qfLogger.level >= qfLoggerLevel) {
                qfLogger.lvlBuild(qfLoggerLevel,
                        record.getSourceMethodName()).add(record.getMessage()).logAt(record.getMillis());
            }
            return passThrough;
        }
    }
    // }}}
    // {{{ javaLoggerLevelToQFLogLevel

    /**
     * Translates a java.util.logging.Level into a Log level
     */
    public static int javaLoggerLevelToQFLogLevel(final java.util.logging.Level level)
    {
        if (java.util.logging.Level.SEVERE.equals(level)) {
            return Log.ERR;
        } else if (java.util.logging.Level.WARNING.equals(level)) {
            return Log.WRN;
        } else if (java.util.logging.Level.INFO.equals(level)) {
            return Log.MSG;
        } else if (java.util.logging.Level.CONFIG.equals(level)) {
            return Log.MTD;
        }
        return Log.DBG;
    }

    // }}}
    // {{{ qfLoggerLevelToJavaLogLevel

    /**
     * Translates a Log level into a java.util.logging.Level
     */
    public static java.util.logging.Level qfLoggerLevelToJavaLogLevel(final int level)
    {
        switch (level) {
        case Log.ERR:
        case Log.ERRDETAIL: return java.util.logging.Level.SEVERE;
        case Log.WRN:
        case Log.WRNDETAIL: return java.util.logging.Level.WARNING;
        case Log.MSG:
        case Log.MSGDETAIL: return java.util.logging.Level.INFO;
        case Log.MTD:
        case Log.MTDDETAIL: return java.util.logging.Level.CONFIG;
        default: return java.util.logging.Level.FINEST;
        }
    }

    // }}}

    //---------------------------------------------------------------------------------------
    // Methods for QFLoggerInterface
    //---------------------------------------------------------------------------------------
    // {{{ getLevel()

    public int getLevel() {
        return level;
    }

    // }}}
    // {{{ log(int,long,java.lang.String,java.lang.String)

    public void log(final int level, final long timestamp, final String method,
            final String message) {
        Log.log(level, timestamp, getOwnerName(), method, message);
    }

    // }}}
}
// }}}
