// {{{ copyright

/********************************************************************
 *
 * The contents of this file are subject to the Mozilla Public
 * License Version 1.1 (the "License"); you may not use this file
 * except in compliance with the License. You may obtain a copy of
 * the License at http://www.mozilla.org/MPL/
 *
 * Software distributed under the License is distributed on an "AS
 * IS" basis, WITHOUT WARRANTY OF ANY KIND, either express or
 * implied. See the License for the specific language governing
 * rights and limitations under the License.
 *
 * The Original Code is qfs.de code.
 *
 * The Initial Developer of the Original Code is Gregor Schmid.
 * Portions created by Gregor Schmid are
 * Copyright (C) 2000 Quality First Software, Gregor Schmid.
 * All Rights Reserved.
 *
 * Contributor(s):
 *
 *******************************************************************/

// }}}

package de.qfs.lib.log;

// {{{ imports

import java.io.File;
import java.io.IOException;

// }}}

// {{{ doc

/**
 * This class is similar to {@link FileLogWriter FileLogWriter} but limits the
 * size of the files it writes to. It keeps a limited number of log files in a
 * kind of ring buffer as follows:<p>
 *
 * The first time the RingFileLogWriter logs to a file, lets call it
 * <code>myfile.log</code>, the file is created. When the size limit is
 * reached, it is renamed to <code>myfile1.log</code> and a new file named
 * <code>myfile.log</code> is created, to which the RingFileLogWriter now
 * sends its output. When this file fills up as well, <code>myfile1.log</code>
 * is renamed to <code>myfile2.log</code> and <code>myfile.log</code> is again
 * renamed to <code>myfile1.log</code>. This continues until the file number
 * limit is reached (or ad infinitum if the limit is negative). Then every
 * time a new log file is created, the oldest log file (the one with the
 * highest number) is removed.
 *
 * @author      Gregor Schmid
 * @since       0.98.0
 */

// }}}
public class RingFileLogWriter
    extends FileLogWriter
{
    // {{{ variables

    /**
     * Default size limit.
     */
    public final static int DEFAULT_SIZE_LIMIT = 1024;

    /**
     * Default file limit.
     */
    public final static int DEFAULT_FILE_LIMIT = 5;

    /**
     * LevelFilter for convenience methods.
     */
    private static LevelFilter lastLevelFilter;

    /**
     * The maximum size of a log file (in kB). Actually files may get quite a
     * bit larger than this, since the size is only checked after writing the
     * whole array of {@link LogEntry LogEntries} in {@link
     * #write(de.qfs.lib.log.LogEntry[]) write}.
     */
    protected int sizeLimit = DEFAULT_FILE_LIMIT;

    /**
     * The maximum number of files to keep around. Set this to -1 to keep an
     * unlimited number of log files.
     */
    protected int fileLimit = DEFAULT_FILE_LIMIT;

    /**
     * The client name, needed for subsequent new log files.
     */
    protected String client;

    /**
     * The basename of the log file.
     */
    protected String basename;

    /**
     * The extension of the log file (including '.').
     */
    protected String ext;

    /**
     * The directory of the log files.
     */
    protected File dir;

    /**
     * Counter for checking size/rotating files.
     */
    private int writeCount;

    // }}}

    //----------------------------------------------------------------------
    // Constructors
    //----------------------------------------------------------------------
    // {{{ RingFileLogWriter(String,String,int,int)

    /**
     * Create a new RingFileLogWriter that uses a {@link DefaultLogFormat
     * DefaultLogFormat} to write {@link LogEntry LogEntries} to a file.
     *
     * @param   client  Name of the client, used by qflog.
     * @param   file    The name of the file to print to. If the file already
     *                  exists, append to it.
     * @param   sizeLimit The maximum size of a log file (in kB).
     * @param   fileLimit The maximum number of old log files to keep around.
     *
     * @throws  IOException     If the file cannot be opened for writing.
     */
    public RingFileLogWriter(String client, String file, int sizeLimit,
                             int fileLimit)
        throws IOException
    {
        super(client, file, MODE_APPEND, true);
        this.client = client;
        this.sizeLimit = sizeLimit;
        this.fileLimit = fileLimit;

        String parent = this.file.getParent();
        if (parent == null) {
            parent = ".";
        }
        dir = new File (parent);

        String name = this.file.getName();
        int pos = name.lastIndexOf('.');
        if (pos > 0) {
            ext = name.substring(pos);
            basename = name.substring(0, pos);
        } else {
            ext = "";
            basename = name;
        }
        checkLimit();
    }

    // }}}
    // {{{ RingFileLogWriter(String,String,int,int,LogFormat)

    /**
     * Create a new RingFileLogWriter that writes {@link LogEntry LogEntries}
     * to a file.
     *
     * @param   client  Name of the client, used by qflog.
     * @param   file    The name of the file to print to. If the file already
     *                  exists, append to it.
     * @param   sizeLimit The maximum size of a log file (in kB).
     * @param   fileLimit The maximum number of old log files to keep around.
     * @param   format  The format used to print LogEntries.
     *
     * @throws  IOException     If the file cannot be opened for writing.
     */
    public RingFileLogWriter(String client, String file, int sizeLimit,
                             int fileLimit, LogFormat format)
        throws IOException
    {
        super(client, file, MODE_APPEND, true, format);
        this.client = client;
        this.sizeLimit = sizeLimit;
        this.fileLimit = fileLimit;

        String parent = this.file.getParent();
        if (parent == null) {
            parent = ".";
        }
        dir = new File (parent);

        String name = this.file.getName();
        int pos = name.lastIndexOf('.');
        if (pos > 0) {
            ext = name.substring(pos);
            basename = name.substring(0, pos);
        } else {
            ext = "";
            basename = name;
        }
        checkLimit();
    }

    // }}}

    //----------------------------------------------------------------------
    // Static convenience methods.
    //----------------------------------------------------------------------
    // {{{ logToFile(String,String,int,int)

    /**
     * Log messages to a log file by creating a LevelFilter with a
     * RingFileLogWriter and adding it to the Log filter chain. The log file
     * will be in the format recognized by the qflog log server.
     *
     * @param   client  Name of the client, used by qflog.
     * @param   file    The name of the file to save in.
     * @param   sizeLimit The maximum size of a log file (in kB).
     * @param   fileLimit The maximum number of old log files to keep around.
     *
     * @return The new LevelFilter. Unless you change its level via {@link
     * de.qfs.lib.log.LevelFilter#setLevel LevelFilter.setLevel}, it will log
     * all messages to the file. If you only call this method once in your
     * application, you can use {@link #stopLogging stopLogging} to remove the
     * filter and close the writer, otherwise you have to save this reference
     * to the filter and clean up yourself.
     *
     * @throws  IOException     If the file cannot be created.
     */
    public static LevelFilter logToFile(String client, String file,
                                        int sizeLimit, int fileLimit)
        throws IOException
    {
        lastLevelFilter = new LevelFilter
            (-1, new RingFileLogWriter (client, file, sizeLimit, fileLimit));
        Log.addFilter(lastLevelFilter);
        return lastLevelFilter;
    }

    // }}}
    // {{{ logToFile(String,String,int,int,LogFormat)

    /**
     * Log messages to a log file by creating a LevelFilter with a
     * RingFileLogWriter and adding it to the Log filter chain. The log file
     * will be in the format recognized by the qflog log server.
     *
     * @param   client  Name of the client, used by qflog.
     * @param   file    The name of the file to save in.
     * @param   sizeLimit The maximum size of a log file (in kB).
     * @param   fileLimit The maximum number of old log files to keep around.
     * @param   format  The format used to print LogEntries.
     *
     * @return The new LevelFilter. Unless you change its level via {@link
     * de.qfs.lib.log.LevelFilter#setLevel LevelFilter.setLevel}, it will log
     * all messages to the file. If you only call this method once in your
     * application, you can use {@link #stopLogging stopLogging} to remove the
     * filter and close the writer, otherwise you have to save this reference
     * to the filter and clean up yourself.
     *
     * @throws  IOException     If the file cannot be created.
     */
    public static LevelFilter logToFile(String client, String file,
                                        int sizeLimit, int fileLimit,
                                        LogFormat format)
        throws IOException
    {
        lastLevelFilter = new LevelFilter
            (-1, new RingFileLogWriter (client, file, sizeLimit, fileLimit,
                                        format));
        Log.addFilter(lastLevelFilter);
        return lastLevelFilter;
    }

    // }}}
    // {{{ stopLogging

    /**
     * Remove the last {@link LevelFilter LevelFilter} instance created with
     * {@link #logToFile logToFile} from the filter chain and close the log
     * file. Use this method if you only call {@link #logToFile logToFile}
     * once in your application.
     */
    public static void stopLogging()
    {
        if (lastLevelFilter != null) {
            lastLevelFilter.getLogWriter().close();
            Log.removeFilter(lastLevelFilter);
            lastLevelFilter = null;
        }
    }

    // }}}

    //----------------------------------------------------------------------
    // Public getters and setters.
    //----------------------------------------------------------------------
    // {{{ getSizeLimit

    /**
     * Get the file size limit (in kB) of the RingFileLogWriter.
     *
     * @return  The sizeLimit of the RingFileLogWriter.
     */
    public final int getSizeLimit()
    {
        return sizeLimit;
    }

    // }}}
    // {{{ setSizeLimit

    /**
     * Set the file size limit (in kB) of the RingFileLogWriter. This causes an
     * immediate check of the current file size.
     *
     * @param   sizeLimit       The sizeLimit to set.
     */
    public synchronized final void setSizeLimit(int sizeLimit)
    {
        this.sizeLimit = sizeLimit;
        checkLimit();
    }

    // }}}
    // {{{ getFileLimit

    /**
     * Get the number of old log files to keep.
     *
     * @return The number of old log files to keep (negative means umlimited)
     */
    public final int getFileLimit()
    {
        return fileLimit;
    }

    // }}}
    // {{{ setFileLimit

    /**
     * Get the number of old log files to keep. A negative value means
     * no limit.
     *
     * @param   fileLimit       The file limit to set.
     */
    public final void setFileLimit(int fileLimit)
    {
        this.fileLimit = fileLimit;
    }

    // }}}

    //----------------------------------------------------------------------
    // The LogWriter interface
    //----------------------------------------------------------------------
    // {{{ write(LogEntry)

    /**
     * Write one LogEntry.
     *
     * @param   entry   The entry to write.
     */
    @Override
    public synchronized void write(LogEntry entry)
    {
        if (closed) {
            return;
        }
        super.write(entry);
        checkLimit();
    }

    // }}}
    // {{{ write(LogEntry[])

    /**
     * Write an array of LogEntires in one go. Clients of the RingFileLogWriter
     * should use this method in preference to {@link #write(LogEntry)
     * write(LogEntry)}, since it is more efficient.
     *
     * @param   entries The entries to write.
     */
    @Override
    public synchronized void write(LogEntry[] entries)
    {
        if (closed) {
            return;
        }
        super.write(entries);
        checkLimit();
    }

    // }}}

    //----------------------------------------------------------------------
    // Helper methods
    //----------------------------------------------------------------------
    // {{{ checkLimit

    /**
     * Check whether the size limit is exceeded and act accordingly.
     */
    protected synchronized void checkLimit()
    {
        if (writeCount++ < 100) {
            return;
        }
        writeCount = 0;
        if (file.length() / 1024 >= sizeLimit) {
            int max = removeFiles();
            moveFiles(max);
            openFile(client, file.getPath(), MODE_CREATE);
        }
    }

    // }}}
    // {{{ removeFiles

    /**
     * Delete files that will exceed the file limit.
     *
     * @return  The maximum number of the files left on disk.
     */
    protected int removeFiles()
    {
        String[] files = dir.list();
        int max = -1;
        for (int i = 0; i < files.length; i++) {
            int num = getFileNum(files[i]);
            if (fileLimit >= 0 && num >= fileLimit) {
                new File (dir, files[i]).delete();
            }
            if (num > max) {
                max = num;
            }
        }
        return max;
    }

    // }}}
    // {{{ moveFiles

    /**
     * Rename files, incrementing their number by 1.
     *
     * @param   max     The maximim number to start at if umlimited files are
     *                  kept.
     */
    protected synchronized void moveFiles(int max)
    {
        int start = fileLimit < 0 ? max : fileLimit - 1;
        for (int i = start; i >= 0; i--) {
            File old = i == 0 ? file : new File (dir, basename + i + ext);
            if (old.exists()) {
                File newFile = new File (dir, basename + (i + 1) + ext);
                if (! old.renameTo(newFile)) {
                    System.err.println("RingFileLogWriter: Failed to rename " + // checkcode:ignore
                                       old.getPath() + " to " + newFile);
                }
            }
        }
    }

    // }}}
    // {{{ getFileNum

    /**
     * Get the number of a file.
     *
     * @param   file    The file to check.
     *
     * @return The files number if it starts with basename and ends with ext
     * or -1 if it doesn't match.
     */
    protected int getFileNum(String file)
    {
        if (file.startsWith(basename)
            && file.endsWith(ext)
            && file.length() > basename.length() + ext.length()) {
            String num = file.substring(basename.length(),
                                        file.length() - ext.length());
            try {
                return Integer.parseInt(num);
            } catch (NumberFormatException ex) {
                return -1;
            }
        }
        return -1;
    }

    // }}}
}
