// {{{ copyright

/********************************************************************
 *
 * The contents of this file are subject to the Mozilla Public
 * License Version 1.1 (the "License"); you may not use this file
 * except in compliance with the License. You may obtain a copy of
 * the License at http://www.mozilla.org/MPL/
 *
 * Software distributed under the License is distributed on an "AS
 * IS" basis, WITHOUT WARRANTY OF ANY KIND, either express or
 * implied. See the License for the specific language governing
 * rights and limitations under the License.
 *
 * The Original Code is qfs.de code.
 *
 * The Initial Developer of the Original Code is Gregor Schmid.
 * Portions created by Gregor Schmid are
 * Copyright (C) 1999 Quality First Software, Gregor Schmid.
 * All Rights Reserved.
 *
 * Contributor(s):
 *
 *******************************************************************/

// }}}

package de.qfs.lib.log;

// {{{ imports

import java.io.File;
import java.io.FileOutputStream;
import java.io.IOException;
import java.io.OutputStream;
import java.io.OutputStreamWriter;
import java.io.PrintWriter;
import java.io.Writer;

import java.text.DateFormat;

import java.util.Vector;

// }}}

// {{{ doc

/**
 * A StreamFilter prints log messages to a stream. Its main use is the creation
 * of log files. <p>
 *
 * The simplest way to create a log file is the convenience method {@link
 * #logToFile logToFile}, that expects a client name for use with the log
 * server qflog, a file and a mode, defining whether to overwrite an existing
 * file, append to it or to create a unique filename first. <p>
 *
 * For combination with other LogFilters, LogStream implements the LogUser
 * interface to customize, which log messages will be passed on. Its default
 * filter mode is {@link LogUser#FILTER_NONE FILTER_NONE}.<p>
 *
 * By setting a maximum log level for the StreamFilter, some control over
 * which messages are printed to the stream is possible. For fine grained
 * control, additional LogFilters can be installed after the StreamFilter.
 *
 * @author      Gregor Schmid

 * @deprecated Replaced by {@link LevelFilter LevelFilter} in combination with
 * {@link StreamLogWriter StreamLogWriter}, {@link FileLogWriter
 * FileLogWriter} or {@link RingFileLogWriter RingFileLogWriter} as of version
 * 0.98.0.
 */

// }}}
public class StreamFilter implements LogUser
{
    // {{{ variables

    /**
     * StreamFilter instance for convenience methods.
     */
    private static StreamFilter instance;

    /**
     * Create a new log file, overwriting existing files.
     */
    public final static int MODE_CREATE = 0;

    /**
     * Append to a log file, creating a new one if necessary.
     */
    public final static int MODE_APPEND = 1;

    /**
     * Create a new log file. A number appended to the basename is incremented
     * until the file name is unique.
     */
    public final static int MODE_UNIQUE = 2;

    /**
     * The output stream.
     */
    private PrintWriter out;

    /**
     * The LogUser mode.
     */
    private int mode = FILTER_NONE;

    /**
     * The maximum level to output.
     */
    private int level = -1;

    /**
     * The format for timestamps.
     */
    private DateFormat format;

    /**
     * The LogFormat used to format LogEntries.
     */
    private LogFormat logFormat;

    // }}}

    // {{{ constructors

    /**
     * Create a new StreamFilter.
     *
     * @param   stream  The Stream to write the log entries to.
     */
    public StreamFilter(OutputStream stream)
    {
        this (new OutputStreamWriter(stream), -1, null);
    }

    /**
     * Create a new StreamFilter.
     *
     * @param   stream  The Stream to write the log entries to.
     * @param   level   Only messages up to this level will be printed to the
     *                  stream.
     */
    public StreamFilter(OutputStream stream, int level)
    {
        this (new OutputStreamWriter(stream), level, null);
    }

    /**
     * Create a new StreamFilter.
     *
     * @param   stream  The Stream to write the log entries to.
     * @param   level   Only messages up to this level will be printed to the
     *                  stream.
     * @param   format  The DateFormat to use when printing LogEntries.
     */
    public StreamFilter(OutputStream stream, int level, DateFormat format)
    {
        this (new OutputStreamWriter(stream), level, format);
    }

    /**
     * Create a new StreamFilter.
     *
     * @param   writer  The Writer to write the log entries to.
     */
    public StreamFilter(Writer writer)
    {
        this (writer, -1, null);
    }

    /**
     * Create a new StreamFilter.
     *
     * @param   writer  The Writer to write the log entries to.
     * @param   level   Only messages up to this level will be printed to the
     *                  writer.
     */
    public StreamFilter(Writer writer, int level)
    {
        this (writer, level, null);
    }

    /**
     * Create a new StreamFilter.
     *
     * @param   writer  The Writer to write the log entries to.
     * @param   level   Only messages up to this level will be printed to the
     *                  writer.
     * @param   format  The DateFormat to use when printing LogEntries.
     */
    public StreamFilter(Writer writer, int level, DateFormat format)
    {
        if (writer instanceof PrintWriter) {
            out = (PrintWriter) writer;
        } else {
            out = new PrintWriter(writer, true);
        }
        this.level = level;
        this.format = format;
        logFormat = format == null ? new DefaultLogFormat ()
            : new DefaultLogFormat (format);
    }

    // }}}

    //----------------------------------------------------------------------
    // Static convenience methods.
    //----------------------------------------------------------------------
    // {{{ logToFile

    /**
     * Log messages to a log file by creating a StreamFilter and adding it to
     * the Log filter chain. The log file will be in the format recognized by
     * the qflog log server.
     *
     * @param client    Name of the client, used by qflog.
     * @param file      The file to save in.
     * @param mode      How the file should be created. Must be one of
     *                  {@link #MODE_CREATE MODE_CREATE}, {@link #MODE_APPEND
     *                  MODE_APPEND}, or {@link #MODE_UNIQUE MODE_UNIQUE}.
     *
     * @return  The new StreamFilter.
     *
     * @throws  IOException     If the file cannot be created.
     *
     * @deprecated Please use {@link FileLogWriter#logToFile
     * FileLogWriter.logToFile} or {@link RingFileLogWriter#logToFile
     * RingFileLogWriter.logToFile} instead.
     */
    public static StreamFilter logToFile(String client, File file, int mode)
        throws IOException
    {
        PrintWriter out;

        switch(mode) {
        case MODE_CREATE:
        case MODE_APPEND:
            out = new PrintWriter
                (new OutputStreamWriter (new FileOutputStream (file.getPath(), mode == MODE_APPEND), "UTF-8"));
            break;
        case MODE_UNIQUE:
            String name = file.getPath();
            String suffix = "";
            int pos = name.lastIndexOf('.');
            if (pos > 0) {
                suffix = name.substring(pos);
                name = name.substring(0, pos);
            }
            for (int id = 1; true; id++) {
                file = new File (name + id + suffix);
                if (! file.exists()) {
                    out = new PrintWriter
                        (new OutputStreamWriter (new FileOutputStream (file), "UTF-8"));
                    break;
                }
            }
            break;
        default:
            throw new IllegalArgumentException
                ("Argument mode must be one of MODE_CREATE, " +
                 "MODE_APPEND or MODE_UNIQUE.");
        }

        if (mode != MODE_APPEND) {
            out.println("client name: " + client);
            out.flush();
        }

        instance = new StreamFilter(out);
        Log.addFilter(instance);
        return instance;
    }

    // }}}
    // {{{ stopLogging

    /**
     * Remove the StreamFilter instance from the Log filter chain and close
     * the log file.
     */
    public static void stopLogging()
    {
        if (instance != null) {
            instance.close();
        }
    }

    // }}}

    //----------------------------------------------------------------------
    // Public methods.
    //----------------------------------------------------------------------
    // {{{ getLogLevel

    /**
     * Get the level limiting the output of messages.
     *
     * @return   Only messages up to the returned level will be printed to the
     *           stream.
     */
    public final int getLogLevel()
    {
        return level;
    }

    // }}}
    // {{{ setLogLevel

    /**
     * Set the level limiting the output of messages.
     *
     * @param   level   Only messages up to this level will be printed to the
     *                  stream.
     */
    public final void setLogLevel(int level)
    {
        this.level = level;
    }

    // }}}
    // {{{ getDateFormat

    /**
     * Get the DateFormat used to print the timestamp of the log messages.
     *
     * @return  The DateFormat used by the StreamFilter.
     */
    public final DateFormat getDateFormat()
    {
        return format;
    }

    // }}}
    // {{{ setDateFormat

    /**
     * Set the DateFormat used to print the timestamp of the log messages.
     *
     * @param   format  The DateFormat to use.
     */
    public final void setDateFormat(DateFormat format)
    {
        this.format = format;
        logFormat = format == null ? new DefaultLogFormat ()
            : new DefaultLogFormat (format);
    }

    // }}}
    // {{{ close

    /**
     * Close the StreamFilter's output stream and remove the StreamFilter from
     * the Log filter chain.
     */
    public void close()
    {
        out.close();
        Log.removeFilter(this);
    }


    // }}}

    //----------------------------------------------------------------------
    // The LogUser interface.
    //----------------------------------------------------------------------
    // {{{ filter(LogEntry)

    /**
     * Take a LogEntry and write it to the stream.
     *
     * @param   logEntry        The entry to write.
     *
     * @return  True if the entry should be passed on to the next filter.
     */
    @Override
    public boolean filter (LogEntry logEntry)
    {
        boolean used = false;
        if (level == -1 || logEntry.getLevel() <= level) {
            out.println(logFormat.format(logEntry));
            out.flush();
            used = true;
        }
        return mode != FILTER_ALL
            && ((used && mode != FILTER_USED)
                || (!used && mode != FILTER_UNUSED));
    }

    // }}}
    // {{{ filter(LogEntry[])

    /**
     * Write all entries from the list to the stream.
     *
     * @param   entries The list of LogEntries.
     */
    @Override
    public LogEntry[] filter (LogEntry[] entries)
    {
        Vector ret = null;
        if (mode != FILTER_ALL
            && mode != FILTER_NONE
            && level != -1) {
            ret = new Vector ();
        }
        boolean mustFlush = false;
        for (int i = 0; i < entries.length; i++) {
            if (level == -1 || entries[i].getLevel() <= level) {
                out.println(logFormat.format(entries[i]));
                if (ret != null && mode == FILTER_UNUSED) {
                    ret.addElement(entries[i]);
                }
                mustFlush = true;
            } else if (ret != null && mode == FILTER_USED) {
                ret.addElement(entries[i]);
            }
        }
        if (mustFlush) {
            out.flush();
        }
        if (mode == FILTER_ALL
            || (ret == null && mode == FILTER_USED)) {
            return new LogEntry[0];
        } else if (mode == FILTER_NONE
                   || (ret == null && mode == FILTER_UNUSED)) {
            return entries;
        } else {
            LogEntry[] result = new LogEntry[ret.size()];
            ret.copyInto(result);
            return result;
        }
    }

    // }}}
    // {{{ setFilterMode

    /**
     * Configure the StreamFilter's filter mode.
     *
     * @param   mode    The mode to set.
     */
    @Override
    public void setFilterMode(int mode)
    {
        this.mode = mode;
    }

    // }}}

}
