// {{{ copyright

/********************************************************************
 *
 * The contents of this file are subject to the Mozilla Public
 * License Version 1.1 (the "License"); you may not use this file
 * except in compliance with the License. You may obtain a copy of
 * the License at http://www.mozilla.org/MPL/
 *
 * Software distributed under the License is distributed on an "AS
 * IS" basis, WITHOUT WARRANTY OF ANY KIND, either express or
 * implied. See the License for the specific language governing
 * rights and limitations under the License.
 *
 * The Original Code is qfs.de code.
 *
 * The Initial Developer of the Original Code is Gregor Schmid.
 * Portions created by Gregor Schmid are
 * Copyright (C) 2000 Quality First Software, Gregor Schmid.
 * All Rights Reserved.
 *
 * Contributor(s):
 *
 *******************************************************************/

// }}}

package de.qfs.lib.log;

// {{{ imports

import java.io.PrintStream;
import java.io.PrintWriter;

// }}}

/**
 * An implementation of the LogWriter interface that writes {@link LogEntry
 * LogEntries} to a {@link java.io.PrintStream PrintStream} or a {@link
 * java.io.PrintWriter PrintWriter}.
 *
 * @author      Gregor Schmid
 * @since       0.98.0
 */
public class StreamLogWriter
    implements LogWriter
{
    // {{{ variables

    /**
     * The format used to print LogEntries.
     */
    protected LogFormat format;

    /**
     * A stream to print to.
     */
    protected PrintStream ps;

    /**
     * A writer to print to.
     */
    protected PrintWriter pw;

    /**
     * Whether the stream or writer must be closed in {@link #close close}.
     */
    protected boolean mustClose;

    /**
     * Whether the StreamLogWriter has been closed.
     */
    protected boolean closed;

    // }}}

    //----------------------------------------------------------------------
    // Constructors
    //----------------------------------------------------------------------
    // {{{ StreamLogWriter()

    /**
     * Create a new StreamLogWriter that uses a {@link DefaultLogFormat
     * DefaultLogFormat} to print {@link LogEntry LogEntries} to System.err.
     */
    public StreamLogWriter()
    {
        format = new DefaultLogFormat ();
    }

    // }}}
    // {{{ StreamLogWriter(LogFormat)

    /**
     * Create a new StreamLogWriter that prints {@link LogEntry LogEntries} to
     * System.err.
     *
     * @param   format  The format used to print LogEntries.
     */
    public StreamLogWriter(LogFormat format)
    {
        this.format = format;
    }

    // }}}
    // {{{ StreamLogWriter(PrintStream,boolean)

    /**
     * Create a new StreamLogWriter that uses a {@link DefaultLogFormat
     * DefaultLogFormat} to print {@link LogEntry LogEntries} to a stream.
     *
     * @param   ps      The stream to print to.
     * @param   own     Whether the stream is owned by the StreamLogWriter
     *                  and should be closed when the StreamLogWriter is
     *                  closed.
     */
    public StreamLogWriter(PrintStream ps, boolean own)
    {
        this.ps = ps;
        mustClose = own;
        format = new DefaultLogFormat ();
    }

    // }}}
    // {{{ StreamLogWriter(PrintStream,boolean,LogFormat)

    /**
     * Create a new StreamLogWriter that prints {@link LogEntry LogEntries} to
     * a stream.
     *
     * @param   ps      The stream to print to.
     * @param   own     Whether the stream is owned by the StreamLogWriter
     *                  and should be closed when the StreamLogWriter is
     *                  closed.
     * @param   format  The format used to print LogEntries.
     */
    public StreamLogWriter(PrintStream ps, boolean own, LogFormat format)
    {
        this.ps = ps;
        mustClose = own;
        this.format = format;
    }

    // }}}
    // {{{ StreamLogWriter(PrintWriter,boolean)

    /**
     * Create a new StreamLogWriter that uses a {@link DefaultLogFormat
     * DefaultLogFormat} to print {@link LogEntry LogEntries} to a writer.
     *
     * @param   pw      The writer to print to.
     * @param   own     Whether the writer is owned by the StreamLogWriter
     *                  and should be closed when the StreamLogWriter is
     *                  closed.
     */
    public StreamLogWriter(PrintWriter pw, boolean own)
    {
        this.pw = pw;
        mustClose = own;
        format = new DefaultLogFormat ();
    }

    // }}}
    // {{{ StreamLogWriter(PrintWriter,boolean,LogFormat)

    /**
     * Create a new StreamLogWriter that prints {@link LogEntry LogEntries} to
     * a writer.
     *
     * @param   pw      The writer to print to.
     * @param   own     Whether the writer is owned by the StreamLogWriter
     *                  and should be closed when the StreamLogWriter is
     *                  closed.
     * @param   format  The format used to print LogEntries.
     */
    public StreamLogWriter(PrintWriter pw, boolean own, LogFormat format)
    {
        this.pw = pw;
        mustClose = own;
        this.format = format;
    }

    // }}}

    //----------------------------------------------------------------------
    // The LogWriter interface
    //----------------------------------------------------------------------
    // {{{ write(LogEntry)

    /**
     * Write one LogEntry.
     *
     * @param   entry   The entry to write.
     */
    @Override
    public void write(LogEntry entry)
    {
        if (closed) {
            return;
        }
        if (ps != null) {
            ps.println(format.format(entry));
            ps.flush();
        } else if (pw != null) {
            pw.println(format.format(entry));
            pw.flush();
        } else {
            System.err.println(format.format(entry)); // checkcode:ignore
        }
    }

    // }}}
    // {{{ write(LogEntry[])

    /**
     * Write an array of LogEntires in one go. Clients of the StreamLogWriter
     * should use this method in preference to {@link #write(LogEntry)
     * write(LogEntry)}, since it is more efficient.
     *
     * @param   entries The entries to write.
     */
    @Override
    public void write(LogEntry[] entries)
    {
        if (closed) {
            return;
        }
        if (ps != null) {
            for (int i = 0; i < entries.length; i++) {
                ps.println(format.format(entries[i]));

            }
            ps.flush();
        } else if (pw != null) {
            for (int i = 0; i < entries.length; i++) {
                pw.println(format.format(entries[i]));
            }
            pw.flush();
        } else {
            for (int i = 0; i < entries.length; i++) {
                System.err.println(format.format(entries[i])); // checkcode:ignore
            }
        }
    }

    // }}}
    // {{{ close

    /**
     * Close the StreamLogWriter. Closes the stream or writer used if it is
     * owned by the StreamLogWriter.
     */
    @Override
    public synchronized void close()
    {
        if (mustClose) {
            if (ps != null) {
                ps.close();
                ps = null;
            } else if (pw != null) {
                pw.close();
                pw = null;
            }
            mustClose = false;
        }
        closed = true;
    }

    // }}}
    // {{{ getFormat

    /**
     * Get the format the StreamLogWriter uses to print {@link LogEntry
     * LogEntries}.
     *
     * @return  The format of the StreamLogWriter.
     */
    @Override
    public final LogFormat getFormat()
    {
        return format;
    }

    // }}}
    // {{{ setFormat

    /**
     * Set the format to use for printing {@link LogEntry LogEntries}.
     *
     * @param   format  The format to set.
     */
    @Override
    public final void setFormat(LogFormat format)
    {
        this.format = format;
    }

    // }}}
}
