// {{{ copyright

/********************************************************************
 *
 * The contents of this file are subject to the Mozilla Public
 * License Version 1.1 (the "License"); you may not use this file
 * except in compliance with the License. You may obtain a copy of
 * the License at http://www.mozilla.org/MPL/
 *
 * Software distributed under the License is distributed on an "AS
 * IS" basis, WITHOUT WARRANTY OF ANY KIND, either express or
 * implied. See the License for the specific language governing
 * rights and limitations under the License.
 *
 * The Original Code is qfs.de code.
 *
 * The Initial Developer of the Original Code is Gregor Schmid.
 * Portions created by Gregor Schmid are
 * Copyright (C) 1999 Quality First Software, Gregor Schmid.
 * All Rights Reserved.
 *
 * Contributor(s):
 *
 *******************************************************************/

// }}}

package de.qfs.lib.log;

// {{{ imports

import java.util.Enumeration;
import java.util.Hashtable;


// }}}

// {{{ doc

/**
 * This class filters log messages based on a tree of log levels.
 * Log levels can be set for methods, classes, packages or even parts
 * thereof. <p>
 *
 * Whether a message is filtered out depends on the most fine grained
 * log level available for the sending class and method.
 * For example if there is a message from class somePackage.SomeClass,
 * method someMethod at level 5 and there is a setting of 4 for somePackage
 * the message will be filtered out. If there is an additional setting of
 * 6 for somePackage.SomeClass.someMethod, the message will be passed on
 * instead.<p>
 *
 * To speed up the decision, log levels are cached in an internal
 * hashtable.
 *
 * @author      Gregor Schmid
 */

// }}}
public class TreeFilter
    extends AbstractLogUser
{
    // {{{ variables

    /**
     * The cache for the log levels.
     */
    protected Hashtable cache = new Hashtable();

    /**
     * The hashtable holding the log levels
     */
    protected Hashtable levels = new Hashtable();

    /**
     * The methods that are already known
     */
    private Hashtable methods = new Hashtable();

    /**
     * The default level for messages that aren't further defined.
     */
    protected int defaultLevel = 2;

    /**
     * Used internally as a separator between class and method names.
     */
    public final static char msep = '-';

    // }}}

    //----------------------------------------------------------------------
    // Constructors
    //----------------------------------------------------------------------
    // {{{ TreeFilter

    /**
     * Create a new TreeFilter with mode {@link LogUser#FILTER_UNUSED
     * FILTER_UNUSED}.
     */
    public TreeFilter ()
    {
        super(FILTER_UNUSED, null);
    }

    // }}}
    // {{{ TreeFilter

    /**
     * Create a new TreeFilter with mode {@link LogUser#FILTER_USED
     * FILTER_USED}.
     *
     * @param   writer  The LogWriter to write useful entries to.
     */
    public TreeFilter (LogWriter writer)
    {
        super(FILTER_USED, writer);
    }

    // }}}
    // {{{ TreeFilter

    /**
     * Create a new TreeFilter.
     *
     * @param   mode    The {@link LogUser LogUser} mode.
     * @param   writer  The LogWriter to write useful entries to.
     */
    public TreeFilter (int mode, LogWriter writer)
    {
        super(mode, writer);
    }

    // }}}

    //----------------------------------------------------------------------
    // Filtering
    //----------------------------------------------------------------------
    // {{{ filter(int, String, String, String)

    /**
     * Decide, whether a message should be logged.
     *
     * @param   level   The level of the message.
     * @param   clazz   The name of the sending class.
     * @param   method  The name of the sending method.
     * @param   message The message.
     *
     * @return  True if the message should be logged.
     */
    public boolean filter(int level, String clazz,
                          String method, String message)
    {
        return level <= getLevel(clazz, method);
    }

    // }}}
    // {{{ isUseful

    /**
     * Decide whether a log message is useful and should be passed to the log
     * writer.
     *
     * @param   entry   The LogEntry to check.
     *
     * @return  True, if the entry is considered useful, false to skip it.
     */
    @Override
    protected boolean isUseful(LogEntry entry)
    {
        return entry.getLevel() <= getLevel(entry.getClazz(),
                                             entry.getMethod());
    }


    // }}}

    //----------------------------------------------------------------------
    // Changing levels
    //----------------------------------------------------------------------
    // {{{ getDefault

    /**
     * Get the default level for unknown classes/methods.
     *
     * @return  The default level.
     */
    public int getDefault ()
    {
        return defaultLevel;
    }

    // }}}
    // {{{ setDefault

    /**
     * Set the default level for unknown classes/methods.
     *
     * @param   defaultLevel    The level to set.
     */
    public void setDefault (int defaultLevel)
    {
        this.defaultLevel = defaultLevel;
        // completely clear the cache
        cache = new Hashtable();
    }

    // }}}
    // {{{ getLevel

    /**
     * Get the level for a message. If the sending class and method were
     * unknown before, call methodAdded on them.
     *
     * @param   clazz   The name of the sending class.
     * @param   method  The originating method.
     *
     * @return  The log level for clazz and method.
     */
    public int getLevel (String clazz, String method)
    {
        Object level = null;
        // First check for cached value
        if (method != null) {
            level = cache.get (clazz + msep + method);
            if (level != null) {
                return ((Integer)level).intValue();
            }
        }

        // Then check for direct hit
        if (method == null) {
            level = levels.get (clazz);
        } else {
            level = levels.get (clazz + msep + method);
        }
        if (level != null) {
            if (method != null) {
                // cache value
                cache.put (clazz + msep + method, level);
            }
            return ((Integer)level).intValue();
        }

        int lvl;

        if (method != null) {
            // check for direct hit on class only
            level = levels.get (clazz);
        }
        if (level != null) {
            lvl = ((Integer)level).intValue();
        } else {
            lvl = getParentLevel(clazz);
        }
        if (method != null) {
            // cache value
            cache.put (clazz + msep + method, Integer.valueOf(lvl));
        }
        // This may have been unknown before
        if (methods.get(clazz + msep + method) == null) {
            methods.put(clazz + msep + method, "");
            methodAdded (clazz, method);
        }
        return lvl;
    }

    // }}}
    // {{{ setLevel

    /**
     * Set the log level for a class and method, where method may be null
     * to set a general value. If recurse is true,  propagate this level
     * to all similar methods.
     *
     * @param   clazz   The name of the class. Should end with '.' to denote
     *                  a package.
     * @param   method  The name of the method. Null for class or package.
     * @param   level   The level to set.
     * @param   recurse If true, set the same level for all further
     *                  detailed classes and their methods, if false
     *                  just clear the cache for the depending methods
     *                  but leave intermediate levels intact.
     */
    public void setLevel (String clazz, String method, int level,
                          boolean recurse)
    {
        Object old;
        String orig = clazz;
        if (clazz.endsWith(".")) {
            clazz = clazz.substring(0, clazz.length() - 1);
        }
        if (method != null) {
            old = levels.get (clazz + msep + method);
            levels.put (clazz + msep + method, Integer.valueOf(level));
            cache.put (clazz + msep + method, Integer.valueOf(level));
        } else {
            old = levels.get (clazz);
            levels.put (clazz, Integer.valueOf(level));
            if (recurse) {
                removeChildren (clazz, level);
            }
            clearCache (orig);
        }
        if (old == null) {
            methodAdded (orig, method);
        }
    }

    // }}}
    // {{{ removeLevel

    /**
     * Unset a level for a class, method pair.
     *
     * @param   clazz   The name of the class. Should end with '.' to denote
     *                  a package.
     * @param   method  The name of the method. Null for class or package.
     * @param   recurse If true, also remove the levels for all further
     *                  detailed classes and their methods, if false
     *                  just clear the cache for the depending messages
     *                  but leave intermediate levels intact.
     */
    public void removeLevel (String clazz, String method, boolean recurse)
    {
        Object old;

        String orig = clazz;
        if (clazz.endsWith(".")) {
            clazz = clazz.substring(0, clazz.length() - 1);
        }
        if (method != null) {
            old = levels.get(clazz + msep + method);
            levels.remove(clazz + msep + method);
            cache.remove(clazz + msep + method);
        } else {
            old = levels.get(clazz);
            levels.remove(clazz);
            if (recurse) {
                removeChildren(clazz, getParentLevel (clazz));
            }
            clearCache(orig);
        }
        // this is probably stupid
        //  if (old == null) {
            //  methodAdded(orig, method);
        //  }

    }

    // }}}

    //----------------------------------------------------------------------
    // Helper methods
    //----------------------------------------------------------------------
    // {{{ getParentLevel

    /**
     * Get the most detailed level available for a class.
     *
     * @param   clazz   The name of the class
     *
     * @return  The level found, or the default level if there is none.
     */
    protected int getParentLevel (String clazz)
    {
        // Remove inner class levels
        String cl = clazz;
        int pos = cl.lastIndexOf ('$');
        while (pos != -1) {
            cl = cl.substring (0, pos);
            Object level = levels.get (cl);
            if (level != null) {
                return ((Integer)level).intValue();
            }
            pos = cl.lastIndexOf ('$');
        }

        // Search outward through packages
        pos = cl.lastIndexOf ('.');
        while (pos != -1) {
            cl = cl.substring (0, pos);
            Object level = levels.get (cl);
            if (level != null) {
                return ((Integer)level).intValue();
            }
            pos = cl.lastIndexOf ('.');
        }

        return defaultLevel;
    }

    // }}}
    // {{{ removeChildren

    /**
     * Remove all further detailed entries for a class.
     *
     * @param   clazz   The name of the class.
     * @param   level   The level to set for leaf entries, i.e. those
     *                  containing a method as well as a class.
     */
    protected void removeChildren (String clazz, int level)
    {
        for (Enumeration keys = levels.keys();
             keys.hasMoreElements();) {
            String name = (String)keys.nextElement();
            if (name.startsWith(clazz)) {
                if (name.indexOf(msep) != -1) {
                    levels.put (name, Integer.valueOf(level));
                } else {
                    levels.remove (name);
                }
            }
        }
    }

    // }}}
    // {{{ clearCache

    /**
     * Remove all cache entries for a class or package.
     *
     * @param   clazz   The name of the class or package.
     */
    protected void clearCache (String clazz)
    {
        for (Enumeration keys = cache.keys();
             keys.hasMoreElements();) {
            String name = (String)keys.nextElement();
            if (name.startsWith(clazz)) {
                cache.remove (name);
            }
        }
    }

    // }}}

    //----------------------------------------------------------------------
    // Notification
    //----------------------------------------------------------------------
    // {{{ methodAdded

    /**
     * This method is intended to be implemented by a possible subclass.
     * It is called, when the log level for a method cannot be directly
     * resolved from the cache.
     *
     * @param   clazz   The new class.
     * @param   method  The new method.
     */
    protected void methodAdded (String clazz, String method)
    {
    }

    // }}}

}
